<?php
session_start();

// Hata raporlamayı aç
ini_set("display_errors", 1);
ini_set("display_startup_errors", 1);
error_reporting(E_ALL);

// CSRF Token Oluşturma
function csrf_token_olustur() {
    if (empty($_SESSION["csrf_token"])) {
        $_SESSION["csrf_token"] = bin2hex(random_bytes(32));
    }
    return $_SESSION["csrf_token"];
}

// CSRF Token Doğrulama
function csrf_token_dogrula($token) {
    if (isset($_SESSION["csrf_token"]) && hash_equals($_SESSION["csrf_token"], $token)) {
        unset($_SESSION["csrf_token"]);
        return true;
    }
    return false;
}

// Metin Temizleme
function temizle($veri) {
    return htmlspecialchars(strip_tags(trim($veri)));
}

// Yedek dosyalarını tara
function get_backup_files() {
    $backup_dir = __DIR__ . '/views/backups/';
    $files = [];
    
    if (is_dir($backup_dir)) {
        $backup_files = glob($backup_dir . '*.sql');
        foreach ($backup_files as $file) {
            $files[] = [
                'name' => basename($file),
                'size' => filesize($file),
                'date' => filemtime($file),
                'path' => $file
            ];
        }
    }
    
    // Tarihe göre sırala (en yeni önce)
    usort($files, function($a, $b) {
        return $b['date'] - $a['date'];
    });
    
    return $files;
}

// Dosya boyutunu formatla
function format_file_size($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

$step = isset($_GET["step"]) ? (int)$_GET["step"] : 1;
$backup_message = $_SESSION["backup_message"] ?? "";
$backup_error = $_SESSION["backup_error"] ?? "";
unset($_SESSION["backup_message"], $_SESSION["backup_error"]);

$csrf_token = csrf_token_olustur();

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (!csrf_token_dogrula($_POST["csrf_token"] ?? "")) {
        $_SESSION["backup_error"] = "Geçersiz CSRF token!";
        header("Location: backup_restore.php");
        exit();
    }

    $step = $_POST["step"] ?? 1;

    if ($step == 1) {
        // Adım 1: Veritabanı Bağlantısı
        $db_server = temizle($_POST["db_server"] ?? "");
        $db_name = temizle($_POST["db_name"] ?? "");
        $db_username = temizle($_POST["db_username"] ?? "");
        $db_password = $_POST["db_password"] ?? "";

        if (empty($db_server) || empty($db_name) || empty($db_username)) {
            $_SESSION["backup_error"] = "Lütfen tüm gerekli alanları doldurun.";
            header("Location: backup_restore.php");
            exit();
        }

        try {
            $pdo = new PDO("mysql:host=" . $db_server . ";dbname=" . $db_name, $db_username, $db_password);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $pdo->exec("SET NAMES utf8");

            $_SESSION["db_config"] = [
                "db_server" => $db_server,
                "db_name" => $db_name,
                "db_username" => $db_username,
                "db_password" => $db_password
            ];

            $_SESSION["backup_message"] = "Veritabanı bağlantısı başarılı! Yedek dosyalarınızı seçin.";
            header("Location: backup_restore.php?step=2");
            exit();

        } catch (Exception $e) {
            $_SESSION["backup_error"] = "Veritabanı bağlantı hatası: " . $e->getMessage();
            header("Location: backup_restore.php");
            exit();
        }
    } elseif ($step == 2) {
        // Adım 2: Yedek Dosyası Seçimi ve Yükleme
        $db_config = $_SESSION["db_config"] ?? null;
        $selected_backup = $_POST["backup_file"] ?? "";

        if (!$db_config || empty($selected_backup)) {
            $_SESSION["backup_error"] = "Veritabanı bilgileri veya yedek dosyası eksik.";
            header("Location: backup_restore.php?step=2");
            exit();
        }

        try {
            $pdo = new PDO("mysql:host=" . $db_config["db_server"] . ";dbname=" . $db_config["db_name"], $db_config["db_username"], $db_config["db_password"]);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $pdo->exec("SET NAMES utf8");

            $backup_path = __DIR__ . '/views/backups/' . $selected_backup;
            
            if (!file_exists($backup_path)) {
                throw new Exception("Seçilen yedek dosyası bulunamadı.");
            }

            $pdo->beginTransaction();

            // Dış anahtar kontrolünü kapat
            $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");

            // SQL dosyasını oku ve çalıştır
            $sql_content = file_get_contents($backup_path);
            if ($sql_content === false) {
                throw new Exception("Yedek dosyası okunamadı.");
            }

            // SQL komutlarını çalıştır
            $pdo->exec($sql_content);

            // Dış anahtar kontrolünü aç
            $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");

            $pdo->commit();

            $_SESSION["backup_message"] = "Yedek dosyası başarıyla yüklendi! Sisteminiz hazır.";
            header("Location: backup_restore.php?step=3");
            exit();

        } catch (Exception $e) {
            if (isset($pdo)) {
                $pdo->rollBack();
            }
            $_SESSION["backup_error"] = "Yedek yükleme hatası: " . $e->getMessage();
            header("Location: backup_restore.php?step=2");
            exit();
        }
    }
}

$backup_files = get_backup_files();
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo - Yedek Yükleme</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            position: relative;
            overflow-x: hidden;
        }
        
        /* Animasyonlu arka plan şekilleri */
        body::before {
            content: '';
            position: fixed;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 1px, transparent 1px);
            background-size: 50px 50px;
            animation: float 20s ease-in-out infinite;
            z-index: -1;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px) rotate(0deg); }
            50% { transform: translateY(-20px) rotate(180deg); }
        }
        
        .main-container {
            max-width: 800px;
            margin: 0 auto;
            padding: 2rem;
            position: relative;
            z-index: 1;
        }
        
        .hero-section {
            text-align: center;
            margin-bottom: 3rem;
            color: white;
        }
        
        .hero-section h1 {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 1rem;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        
        .hero-section p {
            font-size: 1.1rem;
            opacity: 0.9;
            margin-bottom: 2rem;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            border: none;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            overflow: hidden;
            backdrop-filter: blur(10px);
        }
        
        .card-header {
            background: linear-gradient(135deg, #fd7e14, #e83e8c);
            color: white;
            padding: 1.5rem;
            border-bottom: none;
            text-align: center;
        }
        
        .card-header h4 {
            font-weight: 700;
            margin-bottom: 0;
            letter-spacing: 0.5px;
        }
        
        .card-body {
            padding: 2.5rem;
        }
        
        .form-label {
            font-weight: 600;
            color: #343a40;
        }
        
        .form-control {
            border-radius: 10px;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #fd7e14;
            box-shadow: 0 0 0 0.2rem rgba(253, 126, 20, 0.25);
        }
        
        .btn {
            border-radius: 10px;
            font-weight: 600;
            padding: 12px 25px;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #fd7e14, #e83e8c);
            border: none;
        }
        
        .btn-primary:hover {
            background: linear-gradient(135deg, #e8690b, #d63384);
            transform: translateY(-2px);
        }
        
        .btn-success {
            background: linear-gradient(135deg, #28a745, #20c997);
            border: none;
        }
        
        .btn-success:hover {
            background: linear-gradient(135deg, #218838, #1ea085);
            transform: translateY(-2px);
        }
        
        .backup-file-item {
            background: #f8f9fa;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 1rem;
            margin-bottom: 1rem;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .backup-file-item:hover {
            border-color: #fd7e14;
            background: #fff;
            transform: translateY(-2px);
        }
        
        .backup-file-item.selected {
            border-color: #fd7e14;
            background: linear-gradient(135deg, rgba(253, 126, 20, 0.1), rgba(232, 62, 140, 0.1));
        }
        
        .backup-file-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .backup-file-name {
            font-weight: 600;
            color: #333;
        }
        
        .backup-file-details {
            font-size: 0.9rem;
            color: #666;
        }
        
        .alert {
            border-radius: 10px;
            border: none;
        }
        
        .progress-container {
            background-color: #e9ecef;
            border-radius: 10px;
            height: 20px;
            overflow: hidden;
            margin: 1rem 0;
        }
        
        .progress-bar {
            background: linear-gradient(135deg, #fd7e14, #e83e8c);
            height: 100%;
            transition: width 0.6s ease;
        }
        
        .floating-shapes {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: -1;
        }
        
        .shape {
            position: absolute;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 50%;
            animation: float-shape 6s ease-in-out infinite;
        }
        
        .shape:nth-child(1) {
            width: 60px;
            height: 60px;
            top: 20%;
            left: 10%;
            animation-delay: 0s;
        }
        
        .shape:nth-child(2) {
            width: 100px;
            height: 100px;
            top: 60%;
            right: 10%;
            animation-delay: 2s;
        }
        
        .shape:nth-child(3) {
            width: 40px;
            height: 40px;
            bottom: 20%;
            left: 20%;
            animation-delay: 4s;
        }
        
        @keyframes float-shape {
            0%, 100% { transform: translateY(0px) rotate(0deg); }
            50% { transform: translateY(-30px) rotate(180deg); }
        }
        
        @media (max-width: 768px) {
            .hero-section h1 {
                font-size: 2rem;
            }
            
            .main-container {
                padding: 1rem;
            }
            
            .card-body {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <!-- Animasyonlu arka plan şekilleri -->
    <div class="floating-shapes">
        <div class="shape"></div>
        <div class="shape"></div>
        <div class="shape"></div>
    </div>

    <div class="main-container">
        <!-- Hero Section -->
        <div class="hero-section">
            <h1><i class="bi bi-cloud-upload me-3"></i>Yedek Yükleme</h1>
            <p>Önceden oluşturulmuş yedek dosyalarınızı sisteme yükleyin</p>
        </div>

        <div class="card">
            <div class="card-header">
                <h4><i class="bi bi-database me-2"></i>Odjoo Yedek Yükleme Sihirbazı</h4>
            </div>
            <div class="card-body">
                <?php if ($backup_message): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle-fill me-2"></i>
                        <?= $backup_message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($backup_error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle-fill me-2"></i>
                        <?= $backup_error ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <?php if ($step == 1): ?>
                    <!-- Adım 1: Veritabanı Bağlantısı -->
                    <h5 class="card-title mb-4">
                        <i class="bi bi-1-circle-fill text-primary me-2"></i>
                        Veritabanı Bağlantı Bilgileri
                    </h5>
                    <p class="text-muted mb-4">Lütfen yedek dosyasını yükleyeceğiniz veritabanı bilgilerini girin.</p>
                    
                    <form method="post">
                        <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                        <input type="hidden" name="step" value="1">
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="db_server" class="form-label">Veritabanı Sunucusu</label>
                                <input type="text" class="form-control" id="db_server" name="db_server" value="localhost" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="db_name" class="form-label">Veritabanı Adı</label>
                                <input type="text" class="form-control" id="db_name" name="db_name" required>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="db_username" class="form-label">Veritabanı Kullanıcı Adı</label>
                                <input type="text" class="form-control" id="db_username" name="db_username" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="db_password" class="form-label">Veritabanı Şifresi</label>
                                <input type="password" class="form-control" id="db_password" name="db_password">
                            </div>
                        </div>
                        
                        <div class="d-grid">
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="bi bi-arrow-right me-2"></i>Bağlantıyı Test Et ve Devam Et
                            </button>
                        </div>
                    </form>

                <?php elseif ($step == 2): ?>
                    <!-- Adım 2: Yedek Dosyası Seçimi -->
                    <h5 class="card-title mb-4">
                        <i class="bi bi-2-circle-fill text-primary me-2"></i>
                        Yedek Dosyası Seçimi
                    </h5>
                    
                    <?php if (empty($backup_files)): ?>
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle me-2"></i>
                            <strong>Yedek dosyası bulunamadı!</strong>
                            <p class="mb-0 mt-2">Lütfen yedek dosyalarınızı <code>views/backups/</code> klasörüne yükleyin.</p>
                        </div>
                        <a href="backup_restore.php" class="btn btn-secondary">
                            <i class="bi bi-arrow-left me-2"></i>Geri Dön
                        </a>
                    <?php else: ?>
                        <p class="text-muted mb-4">Yüklemek istediğiniz yedek dosyasını seçin. Bu işlem mevcut verilerinizi etkileyebilir.</p>
                        
                        <form method="post" id="backupForm">
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            <input type="hidden" name="step" value="2">
                            
                            <div class="mb-4">
                                <?php foreach ($backup_files as $file): ?>
                                    <div class="backup-file-item" onclick="selectBackupFile('<?= $file['name'] ?>')">
                                        <div class="backup-file-info">
                                            <div>
                                                <div class="backup-file-name"><?= $file['name'] ?></div>
                                                <div class="backup-file-details">
                                                    <i class="bi bi-calendar me-1"></i><?= date('d.m.Y H:i', $file['date']) ?>
                                                    <i class="bi bi-file-earmark me-1 ms-3"></i><?= format_file_size($file['size']) ?>
                                                </div>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="radio" name="backup_file" 
                                                       value="<?= $file['name'] ?>" id="backup_<?= $file['name'] ?>">
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <div class="alert alert-warning">
                                <i class="bi bi-exclamation-triangle me-2"></i>
                                <strong>Uyarı:</strong> Bu işlem mevcut veritabanındaki tüm verileri silecek ve yedek dosyasındaki verilerle değiştirecektir.
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary btn-lg" onclick="return confirm('Bu işlem mevcut verilerinizi silecek. Emin misiniz?')">
                                    <i class="bi bi-cloud-upload me-2"></i>Yedek Dosyasını Yükle
                                </button>
                                <a href="backup_restore.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-arrow-left me-2"></i>Geri Dön
                                </a>
                            </div>
                        </form>
                    <?php endif; ?>

                <?php elseif ($step == 3): ?>
                    <!-- Adım 3: Yükleme Tamamlandı -->
                    <div class="text-center">
                        <div class="mb-4">
                            <i class="bi bi-check-circle-fill text-success" style="font-size: 4rem;"></i>
                        </div>
                        <h3 class="text-success mb-3">Yedek Yükleme Tamamlandı!</h3>
                        <p class="text-muted mb-4">Yedek dosyanız başarıyla yüklendi. Sisteminiz artık kullanıma hazır.</p>
                        
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle me-2"></i>
                            <strong>Bilgi:</strong> Yedek dosyasındaki kullanıcı bilgileriyle sisteme giriş yapabilirsiniz.
                        </div>
                        
                        <div class="d-grid gap-2">
                            <a href="views/login.php" class="btn btn-success btn-lg">
                                <i class="bi bi-box-arrow-in-right me-2"></i>Sisteme Giriş Yap
                            </a>
                            <a href="install.php" class="btn btn-outline-primary">
                                <i class="bi bi-house me-2"></i>Ana Sayfaya Dön
                            </a>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function selectBackupFile(filename) {
            // Tüm radio button'ları temizle
            document.querySelectorAll('input[name="backup_file"]').forEach(radio => {
                radio.checked = false;
            });
            
            // Seçilen dosyanın radio button'ını işaretle
            document.getElementById('backup_' + filename).checked = true;
            
            // Tüm dosya item'larından selected class'ını kaldır
            document.querySelectorAll('.backup-file-item').forEach(item => {
                item.classList.remove('selected');
            });
            
            // Seçilen dosya item'ına selected class'ını ekle
            event.currentTarget.classList.add('selected');
        }
    </script>
</body>
</html> 