<?php
/**
 * Input Sanitizer - Locale Bug Prevention
 * 
 * Prevents re-parsing of formatted money values
 * Ensures data integrity in forms
 * 
 * @package Odjoo
 * @category Security
 * @version 3.0.0
 */
class InputSanitizer
{
    /**
     * Sanitize money input for hidden field
     * 
     * CRITICAL: Prevents locale bug
     * 
     * When displaying money in forms:
     * ❌ WRONG: <input value="123,45 TRY">
     * ✅ RIGHT: <input value="12345"> (kuruş as integer)
     * 
     * @param int $kurus Amount in kuruş
     * @return string Safe value for hidden input
     * 
     * @example
     * <input type="hidden" name="birim_fiyat_kurus" 
     *        value="<?= InputSanitizer::moneyForHiddenField($fiyat) ?>">
     */
    public static function moneyForHiddenField(int $kurus): string
    {
        return (string) $kurus;
    }
    
    /**
     * Sanitize money input for display field
     * 
     * For user-facing input fields
     * 
     * @param int $kurus Amount in kuruş
     * @return string Formatted for display (no currency)
     * 
     * @example
     * <input type="text" name="birim_fiyat" 
     *        value="<?= InputSanitizer::moneyForDisplayField($fiyat) ?>">
     */
    public static function moneyForDisplayField(int $kurus): string
    {
        $tl = $kurus / 100.0;
        return number_format($tl, 2, ',', '');
    }
    
    /**
     * Sanitize money for JSON
     * 
     * For AJAX responses
     * 
     * @param int $kurus Amount in kuruş
     * @return int Raw kuruş value
     * 
     * @example
     * echo json_encode([
     *     'fiyat' => InputSanitizer::moneyForJson($fiyat)
     * ]);
     */
    public static function moneyForJson(int $kurus): int
    {
        return $kurus;
    }
    
    /**
     * Validate money input from form
     * 
     * Prevents injection and malformed data
     * 
     * @param mixed $input User input
     * @return bool Valid?
     */
    public static function validateMoneyInput($input): bool
    {
        if (empty($input)) {
            return true; // Empty is valid (will be 0)
        }
        
        // Allow: numbers, dots, commas, spaces
        $pattern = '/^[\d\s\.,]+$/';
        
        if (!preg_match($pattern, $input)) {
            return false;
        }
        
        // Try to parse
        try {
            require_once __DIR__ . '/MoneyHelper.php';
            MoneyHelper::toKurus($input);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Sanitize for database storage
     * 
     * Ensures only integer kuruş values are stored
     * 
     * @param mixed $input
     * @return int
     * @throws InvalidArgumentException
     */
    public static function sanitizeForDatabase($input): int
    {
        if (!self::validateMoneyInput($input)) {
            throw new InvalidArgumentException("Invalid money input: " . var_export($input, true));
        }
        
        require_once __DIR__ . '/MoneyHelper.php';
        return MoneyHelper::toKurus($input);
    }
}
