<?php

// Bu fonksiyon dosyası, sistem_log_ekle() fonksiyonuna ihtiyaç duyabilir.
// Bu yüzden log_fonksiyonlari.php dosyasını dahil ediyoruz.
// Ancak dikkat: Bu dosya başka bir yerden include edildiğinde, bu bağımlılıklar zaten karşılanmış olabilir.
// Tekrar tanımlama hatası almamak için varlığını kontrol ederek dahil edebiliriz.
if (!function_exists('sistem_log_ekle')) {
    require_once __DIR__ . '/log_fonksiyonlari.php';
}


/**
 * Yeni bir borç/alacak kaydı ekler.
 * Bu fonksiyon hem satış hem de alış işlemleri için kullanılabilir.
 *
 * @param PDO    $db             Veritabanı bağlantısı.
 * @param int    $cari_id        İlgili carinin ID'si (müşteri/tedarikçi).
 * @param string $tur            'satis' veya 'alis'.
 * @param int    $kategori_id    İlgili kategori ID'si.
 * @param float  $tutar          İşlem tutarı (KDV dahil).
 * @param string $aciklama       İşlem açıklaması.
 * @param string $tarih          İşlem tarihi (YYYY-MM-DD).
 * @param string|null $vade_tarihi Vade tarihi (YYYY-MM-DD), boş olabilir.
 * @param bool   $stoklu_islem   Stoklu işlem mi (true/false).
 * @param float  $toplam_kdv     İşlemdeki toplam KDV tutarı.
 * @return int|false Eklenen kaydın ID'si veya hata durumunda false.
 */
function borc_alacak_ekle(
    PDO $db,
    int $sirket_id,
    int $cari_id,
    string $tur,
    int $kategori_id,
    float $tutar,
    string $aciklama,
    string $tarih,
    ?string $vade_tarihi = null,
    bool $stoklu_islem = false,
    float $toplam_kdv = 0.0
): int|false {
    try {
        $sql = "INSERT INTO borclar (sirket_id, cari_id, tur, kategori_id, tutar, aciklama, tarih, vade_tarihi, stoklu_islem, toplam_kdv, durum)
                VALUES (:sirket_id, :cari_id, :tur, :kategori_id, :tutar, :aciklama, :tarih, :vade_tarihi, :stoklu_islem, :toplam_kdv, :durum)";

        $stmt = $db->prepare($sql);

        // 'durum' başlangıçta 'odenmedi' olarak ayarlanır, peşin ödenme durumu ayrı bir fonksiyonla güncellenir.
        $durum = 'odenmedi'; 

        $stmt->execute([
            ':sirket_id' => $sirket_id,
            ':cari_id' => $cari_id,
            ':tur' => $tur,
            ':kategori_id' => $kategori_id,
            ':tutar' => $tutar,
            ':aciklama' => $aciklama,
            ':tarih' => $tarih,
            ':vade_tarihi' => $vade_tarihi,
            ':stoklu_islem' => (int)$stoklu_islem,
            ':toplam_kdv' => $toplam_kdv,
            ':durum' => $durum
        ]);

        $borc_id = $db->lastInsertId();

        return $borc_id;

    } catch (PDOException $e) {
        error_log("borc_alacak_ekle hatası: " . $e->getMessage());
        return false;
    }
}

/**
 * Bir borç/alacak kaydını ödendi olarak işaretler ve ödenen miktarı günceller.
 * Bu fonksiyon, peşin ödemelerde veya kısmi ödemelerde kullanılır.
 *
 * @param PDO   $db           Veritabanı bağlantısı.
 * @param int   $borc_id      Güncellenecek borç/alacak kaydının ID'si.
 * @param float $odenen_tutar Ödenen miktar.
 * @return bool Başarı durumu.
 */
function borc_alacak_odendi_isaretle(PDO $db, int $borc_id, float $odenen_tutar): bool {
    try {
        // Mevcut borç bilgisini al
        $stmt_borc = $db->prepare("SELECT tutar, odenen_tutar FROM borclar WHERE id = ?");
        $stmt_borc->execute([$borc_id]);
        $borc = $stmt_borc->fetch(PDO::FETCH_ASSOC);

        if (!$borc) {
            throw new Exception("Borç kaydı bulunamadı.");
        }

        $yeni_odenen_tutar = $borc['odenen_tutar'] + $odenen_tutar;
        $durum = 'kismen_odendi'; // Varsayılan durum: kısmen ödendi

        if ($yeni_odenen_tutar >= $borc['tutar']) {
            $durum = 'odendi'; // Eğer ödenen miktar toplam tutara eşit veya büyükse 'odendi'
        }

        $sql = "UPDATE borclar SET odenen_tutar = :odenen_tutar, durum = :durum WHERE id = :borc_id";
        $stmt = $db->prepare($sql);
        $stmt->execute([
            ':odenen_tutar' => $yeni_odenen_tutar,
            ':durum' => $durum,
            ':borc_id' => $borc_id
        ]);

        // Loglama
        sistem_log_ekle(
            $db,
            'guncelle',
            'borclar',
            $borc_id,
            $borc, // Eski hali
            ['odenen_tutar' => $yeni_odenen_tutar, 'durum' => $durum], // Yeni hali
            "Borç/alacak kaydı güncellendi (ID: {$borc_id}, Ödenen: {$odenen_tutar})"
        );

        return true;

    } catch (Exception $e) {
        error_log("borc_alacak_odendi_isaretle hatası: " . $e->getMessage());
        return false;
    }
}

?>