<?php
// includes/ayar_fonksiyonlari.php

$site_ayarlari = []; // Bu dizi, ayarları bellekte tutmak için kullanılır

/**
 * Site ayarını veritabanından çeker.
 * Ayarlar henüz yüklenmediyse toplu olarak çekilir ve önbelleğe alınır.
 *
 * @param PDO    $db              Veritabanı bağlantısı.
 * @param string $ayar_adi        Çekilecek ayarın adı.
 * @param mixed  $varsayilan_deger Ayar bulunamazsa döndürülecek varsayılan değer.
 * @param int|null $sirket_id     Ayarın ait olduğu şirket ID'si. Null ise genel ayar.
 * @return mixed Ayar değeri veya varsayılan değer.
 */
function get_site_setting(PDO $db, $ayar_adi, $varsayilan_deger = null, ?int $sirket_id = null) {
    global $site_ayarlari;

    // Şirket ID'sine özel önbellek anahtarı oluştur
    $cache_key = $ayar_adi . '_' . ($sirket_id ?? 'global');

    // Eğer ayar önbellekte varsa, doğrudan döndür
    if (isset($site_ayarlari[$cache_key])) {
        return $site_ayarlari[$cache_key];
    }

    try {
        if ($sirket_id !== null && $sirket_id > 0) {
            // Şirkete özel ayarı çek
            $stmt = $db->prepare("SELECT ayar_degeri FROM ayarlar WHERE ayar_adi = :ayar_adi AND sirket_id = :sirket_id");
            $stmt->bindParam(":ayar_adi", $ayar_adi, PDO::PARAM_STR);
            $stmt->bindParam(":sirket_id", $sirket_id, PDO::PARAM_INT);
        } else {
            // Genel ayarı çek (sirket_id NULL veya 0 olanlar)
            $stmt = $db->prepare("SELECT ayar_degeri FROM ayarlar WHERE ayar_adi = :ayar_adi AND (sirket_id IS NULL OR sirket_id = 0)");
            $stmt->bindParam(":ayar_adi", $ayar_adi, PDO::PARAM_STR);
        }
        $stmt->execute();
        $ayar_degeri = $stmt->fetchColumn();

        if ($ayar_degeri !== false) {
            $site_ayarlari[$cache_key] = $ayar_degeri;
            return $ayar_degeri;
        }
        } catch (PDOException $e) {
        error_log("Site ayarı çekilirken hata oluştu (Ayar: {$ayar_adi}, Şirket ID: {$sirket_id}): " . $e->getMessage());
    }

    // Ayar bulunamazsa veya hata olursa varsayılan değeri döndür ve önbelleğe al
    $site_ayarlari[$cache_key] = $varsayilan_deger;
    return $varsayilan_deger;
}

/**
 * Site ayarını veritabanında günceller veya ekler.
 * Ayar güncellendiğinde veya eklendiğinde önbelleği temizler.
 *
 * @param PDO    $db              Veritabanı bağlantısı.
 * @param string $ayar_adi        Güncellenecek veya eklenecek ayarın adı.
 * @param mixed  $ayar_degeri     Ayarın yeni değeri.
 * @param int|null $sirket_id     Ayarın ait olduğu şirket ID'si. Null ise genel ayar.
 * @return bool Başarı durumu.
 */
function update_site_setting(PDO $db, $ayar_adi, $ayar_degeri, ?int $sirket_id = null) {
    global $site_ayarlari; // Önbelleği temizlemek için

    try {
        if ($sirket_id !== null && $sirket_id > 0) {
            // Şirkete özel ayarı güncelle veya ekle
            $stmt = $db->prepare("INSERT INTO ayarlar (ayar_adi, ayar_degeri, sirket_id) VALUES (:ayar_adi, :ayar_degeri, :sirket_id) ON DUPLICATE KEY UPDATE ayar_degeri = :ayar_degeri_update");
            $stmt->bindParam(":ayar_adi", $ayar_adi, PDO::PARAM_STR);
            $stmt->bindParam(":ayar_degeri", $ayar_degeri, PDO::PARAM_STR);
            $stmt->bindParam(":sirket_id", $sirket_id, PDO::PARAM_INT);
            $stmt->bindParam(":ayar_degeri_update", $ayar_degeri, PDO::PARAM_STR);
        } else {
            // Genel ayarı güncelle veya ekle
            $stmt = $db->prepare("INSERT INTO ayarlar (ayar_adi, ayar_degeri, sirket_id) VALUES (:ayar_adi, :ayar_degeri, 0) ON DUPLICATE KEY UPDATE ayar_degeri = :ayar_degeri_update");
            $stmt->bindParam(":ayar_adi", $ayar_adi, PDO::PARAM_STR);
            $stmt->bindParam(":ayar_degeri", $ayar_degeri, PDO::PARAM_STR);
            $stmt->bindParam(":ayar_degeri_update", $ayar_degeri, PDO::PARAM_STR);
        }
        
        $result = $stmt->execute();
        
        // İlgili önbellek girdisini temizle
        $cache_key = $ayar_adi . '_' . ($sirket_id ?? 'global');
        if (isset($site_ayarlari[$cache_key])) {
            unset($site_ayarlari[$cache_key]);
        }
        // Tüm önbelleği temizle (daha güvenli olur)
        $site_ayarlari = []; 

        return $result;
    } catch (PDOException $e) {
        error_log("Site ayarı güncellenirken hata oluştu (Ayar: {$ayar_adi}, Şirket ID: {$sirket_id}): " . $e->getMessage());
        return false;
    }
}
?>