<?php

function get_exchange_rates_from_tcmb() {
    // ARTIK TCMB KULLANILMIYOR: Veritabanından kurları oku
    // Beklenen dönüş: [ 'USD' => ['buying'=>..., 'selling'=>...], ... , 'TRY' => ['buying'=>1,'selling'=>1] ]
    try {
        global $db;
        if (!isset($db)) {
            throw new Exception('DB bağlantısı yok');
        }
        $rates = [];
        $stmt = $db->query("SELECT para_birimi, alis, satis FROM doviz_kurlari ORDER BY id");
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $pb = strtoupper(trim($row['para_birimi']));
            if ($pb === '' ) continue;
            $alis = (float)$row['alis'];
            $satis = (float)$row['satis'];
            if ($alis > 0 && $satis > 0) {
                $rates[$pb] = [ 'buying' => $alis, 'selling' => $satis ];
            }
        }
        // TRY/TL daima 1
        $rates['TRY'] = $rates['TRY'] ?? ['buying' => 1.0, 'selling' => 1.0];
        $rates['TL'] = $rates['TL'] ?? $rates['TRY'];
        return $rates;
    } catch (Exception $e) {
        error_log('Döviz kuru okuma hatası (DB): ' . $e->getMessage());
        return ['TRY' => ['buying' => 1.0, 'selling' => 1.0]];
    }
}

/**
 * İki para birimi arasındaki dönüşüm kurunu döndürür (1 from_currency kaç to_currency eder).
 * Örnek: get_exchange_rate('USD', 'TRY') -> 1 USD kaç TRY eder.
 * Örnek: get_exchange_rate('TRY', 'USD') -> 1 TRY kaç USD eder.
 * * @param string $from_currency Dönüştürülecek para birimi kodu (örn: USD, EUR, TRY)
 * @param string $to_currency Dönüştürülecek hedef para birimi kodu (örn: USD, EUR, TRY)
 * @return float Dönüşüm kuru. Hata durumunda 0 döndürür.
 */
function get_exchange_rate($from_currency, $to_currency) {
    if ($from_currency === $to_currency) {
        return 1.0;
    }

    $rates = get_exchange_rates_from_tcmb();

    // Kurlar çekilemediyse veya gerekli kurlar listede yoksa hata döndür
    if (empty($rates) || (!isset($rates[$from_currency]) && !in_array($from_currency, ['TRY','TL'], true)) || (!isset($rates[$to_currency]) && !in_array($to_currency, ['TRY','TL'], true))) {
        error_log("Döviz kuru bulunamadı veya TCMB kurları eksik: " . $from_currency . " -> " . $to_currency);
        return 0; 
    }

    // 1 birim from_currency'nin TRY cinsinden değerini bul
    // (Banka, from_currency'yi sizden ALIYORSA, o fiyattan TRY verir.)
    $from_currency_std = ($from_currency === 'TL') ? 'TRY' : $from_currency;
    $to_currency_std = ($to_currency === 'TL') ? 'TRY' : $to_currency;
    $from_value_in_try = ($from_currency_std === 'TRY') ? 1.0 : ($rates[$from_currency_std]['buying'] ?? 0); 

    // 1 birim to_currency'nin TRY cinsinden değerini bul
    // (Banka, to_currency'yi size SATIYORSA, o fiyattan TRY ister.)
    $to_value_in_try = ($to_currency_std === 'TRY') ? 1.0 : ($rates[$to_currency_std]['selling'] ?? 0); 

    // Geçersiz kurları kontrol et (sıfır veya negatif olmamalı)
    if ($from_value_in_try <= 0 || $to_value_in_try <= 0) {
        error_log("Sıfır veya geçersiz döviz kuru tespit edildi: " . $from_currency . " veya " . $to_currency);
        return 0;
    }

    // Dönüşüm kuru hesaplaması: (from_currency'nin TRY değeri) / (to_currency'nin TRY değeri)
    // Bu bize 1 from_currency'nin kaç to_currency ettiğini verir.
    // Örn: (1 USD = 32 TL) / (1 EUR = 35.5 TL) = 1 USD = (32 / 35.5) EUR
    return $from_value_in_try / $to_value_in_try;
}

?>