<?php

// Veritabanı bağlantısı (config.php'den gelecek)
global $db;

// Basit çeviri fonksiyonu (şimdilik sadece anahtarı döndürür)
function __t($key) {
    // Buraya daha sonra dil dosyalarından çeviri mantığı eklenebilir
    $translations = [
        'guvenlik_kontrol_paneli' => 'Güvenlik Kontrolleri Paneli',
        'install_php_dosyasi_basariyla_silindi' => 'install.php dosyası başarıyla silindi.',
        'install_php_dosyasi_silinirken_bir_hata_olustu' => 'install.php dosyası silinirken bir hata oluştu. Dosya izinlerini kontrol edin.',
        'install_php_dosyasi_zaten_bulunmuyor' => 'install.php dosyası zaten bulunmuyor.',
        'bilinmiyor' => 'Bilinmiyor',
        'erisim_yok_veya_izinler_okunamıyor' => 'Erişim yok veya izinler okunamıyor.',
        'lutfen_dosya_izinlerini_kontrol_edin' => 'Lütfen dosya izinlerini kontrol edin. (Önerilen: 0644 veya 0600)',
        'uygun' => 'Uygun',
        'onerilmiyor' => 'Önerilmiyor',
        'guvenlik_icin_0644_veya_0600_olarak_ayarlanmasi_onerilir' => 'Güvenlik için 0644 (sadece okunabilir) veya 0600 (sadece sahibi okuyup yazabilir) olarak ayarlanması önerilir.',
        'https_kullanimi' => 'HTTPS Kullanımı',
        'https_aktif' => 'HTTPS Aktif',
        'siteniz_https_uzerinden_guvenli_bir_sekilde_calisiyor' => 'Siteniz HTTPS üzerinden güvenli bir şekilde çalışıyor.',
        'http_kullaniliyor' => 'HTTP Kullanılıyor',
        'siteniz_su_anda_http_guvenli_olmayan_baglanti_uzerinden_calisiyor' => 'Siteniz şu anda HTTP (güvenli olmayan bağlantı) üzerinden çalışıyor. Hassas verilerin güvenliği için HTTPS kullanmanız şiddetle tavsiye edilir. SSL/TLS sertifikası kurmayı düşünün.',
        'bulunmuyor' => 'Bulunmuyor',
        'install_php_dosyasi_sunucunuzda_bulunmuyor_sistem_guvenli' => 'install.php dosyası sunucunuzda bulunmuyor. Sistem güvenli.',
        'mevcut' => 'Mevcut!',
        'install_php_dosyasi_hala_sunucunuzda_bulunuyor' => 'install.php dosyası hala sunucunuzda bulunuyor. Kurulum tamamlandıysa bu dosya güvenlik riski oluşturabilir ve sunucunuzdan silinmelidir.',
        'hata_gunlugu_durumu' => 'Hata Günlüğü Durumu',
        'durum_bilinmiyor' => 'Durum Bilinmiyor',
        'hata_gunlugu_php_errors_log_dosyasi_bulunmuyor' => 'Hata günlüğü (php_errors.log) dosyası bulunmuyor. PHP hatalarının loglandığından emin olun.',
        'yazilamiyor' => 'Yazılamıyor',
        'hata_gunlugu_dosyasina_yazilamiyor_hatalar_kaydedilemeyebilir' => 'Hata günlüğü dosyasına yazılamıyor. Hatalar kaydedilemeyebilir. İzinlerini kontrol edin (önerilen: 0664).' ,
        'hata_var' => 'Hata Var',
        'hata_gunlugu_dosyasinda_kayitlar_var' => 'Hata günlüğü dosyasında kayıtlar var. Sistemde hatalar olabileceğini gösterir. Günlüğü kontrol edip sorunları gidermeniz önerilir.',
        'temiz' => 'Temiz',
        'hata_gunlugu_temiz_herhangi_bir_sorun_gorunmuyor' => 'Hata günlüğü temiz. Herhangi bir sorun görünmüyor.',
        'dizin_yazilamaz' => 'Dizin Yazılamaz',
        'hata_gunlugu_dizini_includes_yazilabilir_degil' => 'Hata günlüğü dizini (includes/) yazılabilir değil. Otomatik olarak hata günlüğü oluşturulamayabilir.',
        'yedekler_dizini_yazilabilirligi' => 'Yedekler Dizini Yazılabilirliği',
        'dizin_bulunamadi' => 'Dizin Bulunamadı',
        'yedekler_dizini_views_backups_bulunamadi' => 'Yedekler dizini (views/backups/) bulunamadı. Lütfen bu dizini oluşturun.',
        'yazilabilir' => 'Yazılabilir',
        'yedekler_dizini_uygun_veritabani_yedeklemesi_yapilabilir' => 'Yedekler dizini uygun. Veritabanı yedeklemesi yapılabilir.',
        'yedekler_dizini_yazilabilir_degil' => 'Yedekler dizini yazılabilir değil. Veritabanı yedekleri alınamayabilir. İzinlerini kontrol edin (önerilen: 0755 veya 0775).'
    ];
    return $translations[$key] ?? str_replace(['_', '.php'], [' ', ''], $key); // Anahtar bulunamazsa, _ yerine boşluk koy ve .php'yi kaldır
}

/**
 * ============================================
 * KDV HESAPLAMA SİSTEMİ - HELPER FUNCTIONS
 * Production Grade | Zero Float Error
 * ============================================
 */

/**
 * Şirketin KDV hesaplama tipini getir
 * 
 * @param PDO $db Veritabanı bağlantısı
 * @param int $sirket_id Şirket ID
 * @return string 'kdv_haric' veya 'kdv_dahil'
 */
function getKdvHesaplamaTipi($db, $sirket_id) {
    static $cache = [];
    
    if (isset($cache[$sirket_id])) {
        return $cache[$sirket_id];
    }
    
    try {
        $stmt = $db->prepare("SELECT ayar_degeri FROM ayarlar 
                              WHERE ayar_adi = 'kdv_hesaplama_tipi' 
                              AND sirket_id = ? LIMIT 1");
        $stmt->execute([$sirket_id]);
        $tip = $stmt->fetchColumn() ?: 'kdv_haric';
        $cache[$sirket_id] = $tip;
        return $tip;
    } catch (PDOException $e) {
        error_log("KDV tipi getirme hatası: " . $e->getMessage());
        return 'kdv_haric';
    }
}

/**
 * KDV oranını yüzdeden basis point'e çevir
 * 
 * @param float $percent Yüzde (örn: 20.0 = %20)
 * @return int Basis points (örn: 2000)
 */
function percentToBasis($percent) {
    return (int) round($percent * 100);
}

/**
 * Basis point'i yüzdeye çevir
 * 
 * @param int $basis Basis points
 * @return float Yüzde
 */
function basisToPercent($basis) {
    return $basis / 100.0;
}

/**
 * Para formatla - Kuruştan TL'ye çevir ve formatla
 * 
 * @param int $kurus Kuruş cinsinden
 * @param string $currency Para birimi
 * @return string Formatlanmış tutar
 */
function formatMoney($kurus, $currency = 'TRY') {
    require_once __DIR__ . '/../classes/MoneyHelper.php';
    return MoneyHelper::format($kurus, $currency);
}

/**
 * Form inputundan kuruş değeri al
 * 
 * @param mixed $input Kullanıcı girişi
 * @return int Kuruş cinsinden
 */
function parseMoneyInput($input) {
    require_once __DIR__ . '/../classes/MoneyHelper.php';
    return MoneyHelper::parseInput($input);
}

/**
 * Kuruştan TL'ye çevir
 * 
 * @param int $kurus Kuruş
 * @return float TL
 */
function kurusToTL($kurus) {
    require_once __DIR__ . '/../classes/MoneyHelper.php';
    return MoneyHelper::toTL($kurus);
}

/**
 * TL'den kuruşa çevir
 * 
 * @param mixed $tl TL
 * @return int Kuruş
 */
function tlToKurus($tl) {
    require_once __DIR__ . '/../classes/MoneyHelper.php';
    return MoneyHelper::toKurus($tl);
}

// CSRF Token Oluşturma
function csrf_token_olustur() {
    if (empty($_SESSION["csrf_token"])) {
        $_SESSION["csrf_token"] = bin2hex(random_bytes(32));
    }
    return $_SESSION["csrf_token"];
}

// CSRF Token Doğrulama
function csrf_token_dogrula($token) {
    if (isset($_SESSION["csrf_token"]) && hash_equals($_SESSION["csrf_token"], $token)) {
        unset($_SESSION["csrf_token"]);
        return true;
    }
    return false;
}

// Güvenli Çıkış
function cikis_yap() {
    session_unset();
    session_destroy();
    header("Location: login.php");
    exit();
}

// Yetki Kontrolü
function yetkili_mi($yetki_adlari = null, $yonlendir = true) {
    // Mevcut sayfanın adını al (örn: index.php)
    $mevcut_sayfa = basename($_SERVER['PHP_SELF']);
    
    // Şirketi olmayan yeni kullanıcıların erişebileceği istisnai sayfalar
    $istisna_sayfalar = ['index.php', 'sirket_olustur.php', 'cikis.php'];

    // Önce giriş yapılmış mı diye kontrol et
    if (!isset($_SESSION['kullanici_id'])) {
        if ($yonlendir) {
            header("Location: " . BASE_URL . "/views/login.php");
        exit();
    }
        return false;
    }

    // Eğer kullanıcının şirketi yoksa...
    if (empty($_SESSION['kullanici_sirketleri']) || empty($_SESSION['aktif_sirket_id'])) {
        // ... ve bulunduğu sayfa istisnalardan biri DEĞİLSE, şirket oluşturmaya yönlendir.
        if (!in_array($mevcut_sayfa, $istisna_sayfalar)) {
             if ($yonlendir) {
                header("Location: " . BASE_URL . "/views/index.php"); // index.php'ye yönlendir, o da zaten uyarı gösterecek.
        exit();
    }
            return false;
        }
        // Şirketi yok ama istisnai bir sayfada, devam etmesine izin ver.
        return true;
    }

    // Buradan sonrası, şirketi olan kullanıcılar için normal yetki kontrolü
    if (empty($yetki_adlari)) {
        return true; // Sadece giriş yapmış olması yeterli
    }

    if (isset($_SESSION['yetki_seviyesi']) && $_SESSION['yetki_seviyesi'] === 'admin') {
        return true;
    }

    $kullanici_yetkiler = isset($_SESSION['yetkiler']) ? json_decode($_SESSION['yetkiler'], true) : [];
    if (!is_array($kullanici_yetkiler)) {
        $kullanici_yetkiler = [];
    }

    // Gelen parametre dizi değilse, diziye çevir
    if (!is_array($yetki_adlari)) {
        $yetki_adlari = [$yetki_adlari];
    }

    // Yetki adları dizisindeki herhangi bir yetki kullanıcıda var mı?
    $yetki_bulundu = false;
    foreach ($yetki_adlari as $yetki) {
        if (isset($kullanici_yetkiler[$yetki]) && $kullanici_yetkiler[$yetki]) {
            $yetki_bulundu = true;
            break; // Bir tane bile yetki bulunursa döngüden çık
        }
    }

    if (!$yetki_bulundu && $yonlendir) {
        $_SESSION['hata'] = "Bu sayfaya erişim yetkiniz bulunmamaktadır.";
        header("Location: " . BASE_URL . "/views/index.php");
            exit();
        }

    return $yetki_bulundu;
}


// Yetki Seviyesi Rozeti
function yetki_seviyesi_badge($yetki_seviyesi) {
    switch ($yetki_seviyesi) {
        case "admin":
            return 
                "<span class=\"badge bg-danger\">Admin</span>";
        case "yonetici":
            return 
                "<span class=\"badge bg-warning text-dark\">Yönetici</span>";
        case "kullanici":
            return 
                "<span class=\"badge bg-info text-dark\">Kullanıcı</span>";
        case "misafir":
            return 
                "<span class=\"badge bg-secondary\">Misafir</span>";
        default:
            return 
                "<span class=\"badge bg-light text-dark\">Bilinmiyor</span>";
    }
}

// Güvenli Çıkış (Brute Force Koruması için)
function brute_force_reset() {
    if (isset($_SESSION["login_attempts"])) {
        unset($_SESSION["login_attempts"]);
    }
    if (isset($_SESSION["last_login_attempt"])) {
        unset($_SESSION["last_login_attempt"]);
    }
}

// Metin Temizleme
function temizle($data) {
    if(is_array($data)) {
        return array_map("temizle", $data);
    }
    return htmlspecialchars(trim((string)$data), ENT_QUOTES | ENT_SUBSTITUTE, "UTF-8");
}

// Navbar Menü Filtreleme
function navbar_menu_filtrele() {
    $menu_items = [
        'islemler' => [
            'satis_yeni' => __('sales_invoice'),
            'alis_yeni' => __('purchase_invoice'),
            'cari_alacaklandir' => __('credit_customer'),
            'cari_borclandir' => __('debit_customer'),
            'odeme_al' => __('receive_payment'),
            'odeme_yap' => __('make_payment'),
            'transfer_ekle' => __('transfer')
        ],
        'faturalar' => [
            'satis_faturalari' => __('sales_invoices'),
            'alis_faturalari' => __('purchase_invoices')
        ],
        'teklifler' => __('offers'),
        'stok' => [
            'urunler' => __('product_list'),
            'urun_ekle' => __('add_product'),
            'stok_hareketleri' => __('stock_movements'),
            'stok_raporlari' => __('detailed_reports'),
            'en_cok_satan_urunler' => __('best_selling_products')
        ],
        'cariler' => __('customers'),
        'raporlar' => [
            'hesap_hareketleri' => __('account_transactions'),
            'cari_ekstre' => __('customer_statement'),
            'kasa_gider_makbuzlari' => __('cash_expense_receipts'),
            'raporlar' => __('summary_reports')
        ],
        'araclar' => [
            'arama' => __('search'),
            'toplu_islem' => __('batch_operations'),
            'hatirlatici' => __('reminders'),
            'yedek_islemi' => __('backup_operations')
        ],
        'yonetim' => [
            'kullanicilar' => __('users'),
            'loglar' => __('logs'),
            'ayarlar' => __('settings'),
            'guvenlik_kontrol' => __('security_checks'),
            'kategoriler' => __('categories'),
            'hesaplar' => __('accounts')
        ]
    ];
    
    $filtered_menu = [];
    
    foreach ($menu_items as $menu_key => $menu_item) {
        if (is_array($menu_item)) {
            $filtered_submenu = [];
            foreach ($menu_item as $page => $title) {
                if ($page === 'ayarlar' || $page === 'kategoriler' || $page === 'hesaplar') {
                    if (sayfa_yetki_kontrol($page, 'goruntule')) {
                         $filtered_submenu[$page] = $title;
                    }
                } elseif (sayfa_yetki_kontrol($page)) {
                    $filtered_submenu[$page] = $title;
                }
            }
            if (!empty($filtered_submenu)) {
                $filtered_menu[$menu_key] = $filtered_submenu;
            }
        } else {
            // Teklifler gibi tekil menü elemanları için kontrol
            if (sayfa_yetki_kontrol($menu_key)) {
                $filtered_menu[$menu_key] = $menu_item;
            }
        }
    }
    
    return $filtered_menu;
}

/**
 * İşlem butonu oluştur (yetki kontrolü ile)
 * @param string $sayfa_adi Sayfa adı
 * @param string $islem_tipi İşlem türü
 * @param string $url URL
 * @param string $text Buton metni
 * @param string $class CSS sınıfı
 * @param string $icon İkon sınıfı
 * @return string HTML buton veya boş string
 */
function yetki_buton($sayfa_adi, $islem_tipi, $url, $text, $class = 'btn-primary', $icon = '') {
    if (!sayfa_yetki_kontrol($sayfa_adi, $islem_tipi)) {
        return '';
    }
    
    $icon_html = $icon ? "<i class=\"{$icon}\"></i> " : '';
    return "<a href=\"{$url}\" class=\"btn {$class}\">{$icon_html}{$text}</a>";
}

	function getFirmInformation() {
    global $db; 
    $firmInfo = [];

    // Oturumda aktif şirket ID'si var mı kontrol et
    if (isset($_SESSION['aktif_sirket_id'])) {
        $aktif_sirket_id = $_SESSION['aktif_sirket_id'];
        
        $stmt = $db->prepare("SELECT * FROM sirketler WHERE id = ?");
        $stmt->execute([$aktif_sirket_id]);
        $query = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($query) {
            $firmInfo = $query;
        }
    }
    return $firmInfo;
	}

	// Firma bilgilerini global bir değişkene atama
	$firmaBilgileri = getFirmInformation();
		
	function getFirmaAdi() {
		global $firmaBilgileri;
		// 'firma_adi' 'sirket_adi' olarak değiştirildi
		return isset($firmaBilgileri['sirket_adi']) ? $firmaBilgileri['sirket_adi'] : 'Firma Adı Bilinmiyor';
	}

	function getFirmaPara() {
	    global $db;
	    if (isset($_SESSION['aktif_sirket_id'])) {
	        $stmt = $db->prepare("SELECT para_birimi FROM sirketler WHERE id = ?");
	        $stmt->execute([$_SESSION['aktif_sirket_id']]);
	        $para_birimi_kodu = $stmt->fetchColumn();
	        return $para_birimi_kodu ?: 'TRY'; // Bulamazsa varsayılan TRY
	    }
	    return 'TRY'; // Session yoksa varsayılan TRY
	}

	function getParaBirimiSembolu($para_birimi_kodu = null, $sembol_mi = true) {
	    if ($para_birimi_kodu === null) {
	        $para_birimi_kodu = getFirmaPara();
	    }
	    
	    // Eğer sembol_mi false ise, para birimi kodunu döndür
	    if (!$sembol_mi) {
	        return $para_birimi_kodu;
	    }
	    
	    $semboller = [
	        'TRY' => '₺',
	        'USD' => '$',
	        'EUR' => '€',
	        'GBP' => '£'
	    ];

	    return $semboller[$para_birimi_kodu] ?? $para_birimi_kodu; // Sembol bulunamazsa kodu döndür
	}
	
	function getFirmaDil() {
		global $firmaBilgileri;
		return isset($firmaBilgileri['dil']) ? $firmaBilgileri['dil'] : 'Default Dil belirlenemdi';
	}


	function getFirmaLogoYolu() {
		global $firmaBilgileri;
		return isset($firmaBilgileri['logo_yolu']) ? $firmaBilgileri['logo_yolu'] : 'assets/images/placeholder_logo.png'; // Varsayılan bir logo yolu
	}

/**
 * Oturum (Session) ile ilgili güvenlik kontrollerini yapar ve rapor döndürür.
 * @return array Oturum güvenlik durumu ve önerileri içeren dizi.
 */
function session_guvenlik_kontrol() {
    $sonuclar = [
        'session_regenerate_id' => ['durum' => false, 'mesaj' => 'Oturum ID yenilemesi aktif değil.', 'skor' => 0],
        'ip_check' => ['durum' => false, 'mesaj' => 'IP adresi kontrolü yapılmıyor veya uyuşmuyor.', 'skor' => 0],
        'user_agent_check' => ['durum' => false, 'mesaj' => 'User-Agent kontrolü yapılmıyor veya uyuşmuyor.', 'skor' => 0],
        'idle_timeout' => ['durum' => false, 'mesaj' => 'Boşta kalma süresi kontrolü aktif değil.', 'skor' => 0],
        'cookie_httponly' => ['durum' => false, 'mesaj' => 'httponly bayrağı aktif değil.', 'skor' => 0],
        'cookie_secure' => ['durum' => false, 'mesaj' => 'secure bayrağı aktif değil.', 'skor' => 0],
        'cookie_samesite' => ['durum' => false, 'mesaj' => 'samesite bayrağı aktif değil.', 'skor' => 0],
    ];

    // 1. Session ID Yenileme (Session Fixation)
    if (isset($_SESSION['last_regenerate']) && (time() - $_SESSION['last_regenerate'] <= 300)) {
        $sonuclar['session_regenerate_id'] = ['durum' => true, 'mesaj' => 'Oturum ID periyodik olarak yenileniyor.', 'skor' => 20];
    }

    // 2. IP ve User-Agent Kontrolü (Session Hijacking)
    if (isset($_SESSION['HTTP_USER_AGENT']) && ($_SESSION['HTTP_USER_AGENT'] === ($_SERVER['HTTP_USER_AGENT'] ?? ''))) {
        $sonuclar['user_agent_check'] = ['durum' => true, 'mesaj' => 'User-Agent kontrolü aktif ve uyumlu.', 'skor' => 15];
    } else {
        $sonuclar['user_agent_check']['mesaj'] = 'User-Agent kontrolü başarısız veya aktif değil.';
    }
    
    // IP adresini kontrol etmek daha agresif olabilir, bazı ISP'ler IP değiştirir
    // if (isset($_SESSION['REMOTE_ADDR']) && ($_SESSION['REMOTE_ADDR'] === ($_SERVER['REMOTE_ADDR'] ?? ''))) {
    //     $sonuclar['ip_check'] = ['durum' => true, 'mesaj' => 'IP adresi kontrolü aktif ve uyumlu.', 'skor' => 10];
    // } else {
    //     $sonuclar['ip_check']['mesaj'] = 'IP adresi kontrolü başarısız veya aktif değil.';
    // }

    // 3. Boşta Kalma Süresi (Idle Timeout)
    $session_timeout = ini_get('session.gc_maxlifetime'); // PHP ayarından al
    if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] < $session_timeout)) {
        $sonuclar['idle_timeout'] = ['durum' => true, 'mesaj' => 'Boşta kalma süresi kontrolü aktif.', 'skor' => 15];
    } else {
        $sonuclar['idle_timeout']['mesaj'] = 'Boşta kalma süresi kontrolü pasif veya oturum geçmiş.';
    }

    // 4. Çerez Güvenliği (XSS ve CSRF)
    $params = session_get_cookie_params();
    if ($params['httponly']) {
        $sonuclar['cookie_httponly'] = ['durum' => true, 'mesaj' => 'httponly bayrağı aktif.', 'skor' => 10];
    }
    if ($params['secure'] && (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')) {
        $sonuclar['cookie_secure'] = ['durum' => true, 'mesaj' => 'secure bayrağı aktif (HTTPS üzerinde).', 'skor' => 10];
    } elseif ($params['secure'] && !(isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')) {
        $sonuclar['cookie_secure'] = ['durum' => false, 'mesaj' => 'secure bayrağı aktif ancak HTTPS kullanılmıyor.', 'skor' => 0];
    }
    if (isset($params['samesite']) && !empty($params['samesite']) && strtolower($params['samesite']) !== 'none') { // None harici herhangi bir SameSite değeri
        $sonuclar['cookie_samesite'] = ['durum' => true, 'mesaj' => 'samesite bayrağı aktif.', 'skor' => 10];
    }

    $total_score = 0;
    $total_checks = count($sonuclar);
    foreach ($sonuclar as $check) {
        $total_score += $check['skor'];
    }

    return [
        'kontroller' => $sonuclar,
        'skor' => $total_score,
        'toplam_kontrol' => $total_checks
    ];
}

/**
 * Oturum güvenlik kontrol sonuçlarını HTML olarak biçimlendirir.
 * @return string HTML raporu.
 */
function session_guvenlik_html_rapor() {
    global $db; // Veritabanı bağlantısını kullan (gerekirse)
    $rapor = session_guvenlik_kontrol();
    $html = '<div class="card">';
    $html .= '<div class="card-header"><h3 class="card-title"><i class="bi bi-shield-shaded me-2 text-primary"></i>Oturum Güvenlik Detayları</h3></div>';
    $html .= '<div class="card-body p-0">';
    $html .= '<div class="table-responsive">';
    $html .= '<table class="table table-vcenter card-table">';
    $html .= '<thead><tr><th>Kontrol</th><th>Durum</th><th>Mesaj</th><th>Skor</th></tr></thead>';
    $html .= '<tbody>';

    foreach ($rapor['kontroller'] as $kontrol_adi => $veri) {
        $status_icon = $veri['durum'] ? '<i class="bi bi-check-circle-fill text-success"></i>' : '<i class="bi bi-x-circle-fill text-danger"></i>';
        $status_text = $veri['durum'] ? 'Aktif' : 'Pasif';
        $html .= '<tr>';
        $html .= '<td>' . htmlspecialchars(str_replace(['_', 'check'], [' ', ''], $kontrol_adi)) . '</td>';
        $html .= '<td>' . $status_icon . ' ' . $status_text . '</td>';
        $html .= '<td>' . htmlspecialchars($veri['mesaj']) . '</td>';
        $html .= '<td><span class="badge bg-primary">' . $veri['skor'] . '</span></td>';
        $html .= '</tr>';
    }

    $html .= '</tbody>';
    $html .= '<tfoot>';
    $html .= '<tr>';
    $html .= '<th colspan="3" class="text-end">Toplam Güvenlik Skoru:</th>';
    $html .= '<th><span class="badge bg-success fs-4">' . $rapor['skor'] . '</span> / <span class="badge bg-secondary fs-4">' . ($rapor['toplam_kontrol'] * 20) . '</span></th>';
    $html .= '</tr>';
    $html .= '</tfoot>';
    $html .= '</table>';
    $html .= '</div>';
    $html .= '</div>';
    $html .= '</div>';
    return $html;
}

/**
 * Oturum güvenliğini artırmak için PHP ini ayarlarını ve oturum değişkenlerini günceller.
 * Bu işlem mevcut oturumu sonlandırır ve yeni bir oturum başlatmayı gerektirir.
 */
function session_guvenlik_iyilestir() {
    // Mevcut oturumu güvenli bir şekilde sonlandır
    if (session_status() === PHP_SESSION_ACTIVE) {
        session_unset();
        session_destroy();
    }

    // Yeni güvenli ayarlar yap
    ini_set('session.use_strict_mode', 1);
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'));
    ini_set('session.cookie_samesite', 'Lax'); // 'Lax' veya 'Strict' önerilir
    
    // Oturumu tekrar başlat ve yeni güvenlik bilgilerini kaydet
    session_start();
    
    // Session regenerate işlemini güvenli şekilde yap
    if (!headers_sent()) {
        session_regenerate_id(true);
    }
    
    $_SESSION['last_regenerate'] = time();
    $_SESSION['last_activity'] = time();
    $_SESSION['HTTP_USER_AGENT'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
    // $_SESSION['REMOTE_ADDR'] = $_SERVER['REMOTE_ADDR'] ?? ''; // IP kontrolü çok agresif olabilir.

    return true;
}

/**
 * Byte boyutunu insan tarafından okunabilir bir formatta döndürür.
 * @param int $size Byte cinsinden boyut.
 * @param int $precision Ondalık hassasiyeti.
 * @return string Biçimlendirilmiş boyut.
 */
function formatBytes($size, $precision = 2) {
    $base = log($size, 1024);
    $suffixes = array('B', 'KB', 'MB', 'GB', 'TB');
    return round(pow(1024, $base - floor($base)), $precision) . ' ' . $suffixes[floor($base)];
}
?>