<?php
/**
 * MIGRATION SCRIPT: Decimal → Integer (Kuruş) - AUTO MODE
 * 
 * Bu script tüm para değerlerini decimal'den integer'a çevirir
 * 
 * UYARI: Mutlaka yedek alın!
 * 
 * @package Odjoo
 * @version 3.0.0
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/classes/MoneyHelper.php';
require_once __DIR__ . '/classes/TaxEngine.php';

set_time_limit(0);
ini_set('memory_limit', '512M');

echo "==========================================================\n";
echo "  ODJOO - DECIMAL TO INTEGER MIGRATION (AUTO MODE)\n";
echo "  Production-Grade KDV Sistemi v3.0\n";
echo "==========================================================\n\n";

echo "AUTO-CONFIRM MODE: Migration başlatılıyor...\n\n";

try {
    $db->beginTransaction();
    
    // =====================================================
    // PHASE 1: YENİ SÜTUNLAR EKLE
    // =====================================================
    echo "PHASE 1: Yeni sütunlar ekleniyor...\n";
    
    // urunler
    echo "  - urunler tablosu...\n";
    $db->exec("ALTER TABLE urunler
        ADD COLUMN IF NOT EXISTS alis_fiyati_kurus BIGINT AFTER alis_fiyati,
        ADD COLUMN IF NOT EXISTS satis_fiyati_kurus BIGINT AFTER satis_fiyati,
        ADD COLUMN IF NOT EXISTS kdv_orani_basis SMALLINT AFTER kdv_orani");
    
    // fatura_detaylari
    echo "  - fatura_detaylari tablosu...\n";
    $db->exec("ALTER TABLE fatura_detaylari
        ADD COLUMN IF NOT EXISTS birim_fiyat_kurus BIGINT AFTER birim_fiyat,
        ADD COLUMN IF NOT EXISTS kdv_tutari_kurus BIGINT AFTER kdv_tutari,
        ADD COLUMN IF NOT EXISTS toplam_tutar_kurus BIGINT AFTER toplam_tutar,
        ADD COLUMN IF NOT EXISTS kdv_orani_basis SMALLINT AFTER kdv_orani");
    
    // borclar
    echo "  - borclar tablosu...\n";
    $db->exec("ALTER TABLE borclar
        ADD COLUMN IF NOT EXISTS tutar_kurus BIGINT AFTER tutar,
        ADD COLUMN IF NOT EXISTS toplam_kdv_kurus BIGINT AFTER toplam_kdv,
        ADD COLUMN IF NOT EXISTS indirim_tutari_kurus BIGINT AFTER indirim_tutari,
        ADD COLUMN IF NOT EXISTS odenen_tutar_kurus BIGINT AFTER odenen_tutar,
        ADD COLUMN IF NOT EXISTS tax_snapshot LONGTEXT COMMENT 'KDV hesaplama audit trail (JSON)'");
    
    // odemeler
    echo "  - odemeler tablosu...\n";
    $db->exec("ALTER TABLE odemeler
        ADD COLUMN IF NOT EXISTS tutar_kurus BIGINT AFTER tutar");
    
    // stok_hareketleri
    echo "  - stok_hareketleri tablosu...\n";
    $db->exec("ALTER TABLE stok_hareketleri
        ADD COLUMN IF NOT EXISTS birim_fiyat_kurus BIGINT AFTER birim_fiyat");
    
    echo "✓ Yeni sütunlar eklendi\n\n";
    
    // =====================================================
    // PHASE 2: VERİ DÖNÜŞÜMÜ
    // =====================================================
    echo "PHASE 2: Veri dönüşümü yapılıyor...\n";
    
    // urunler
    echo "  - urunler dönüştürülüyor...\n";
    $stmt = $db->query("SELECT id, alis_fiyati, satis_fiyati, kdv_orani FROM urunler");
    $updateStmt = $db->prepare("UPDATE urunler SET 
        alis_fiyati_kurus = ?, 
        satis_fiyati_kurus = ?,
        kdv_orani_basis = ?
        WHERE id = ?");
    
    $count = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $alisFiyatiKurus = MoneyHelper::toKurus($row['alis_fiyati']);
        $satisFiyatiKurus = MoneyHelper::toKurus($row['satis_fiyati']);
        $kdvBasis = TaxEngine::percentToBasis((float)$row['kdv_orani']);
        
        $updateStmt->execute([$alisFiyatiKurus, $satisFiyatiKurus, $kdvBasis, $row['id']]);
        $count++;
    }
    echo "    → $count ürün dönüştürüldü\n";
    
    // fatura_detaylari
    echo "  - fatura_detaylari dönüştürülüyor...\n";
    $stmt = $db->query("SELECT id, birim_fiyat, kdv_tutari, toplam_tutar, kdv_orani FROM fatura_detaylari");
    $updateStmt = $db->prepare("UPDATE fatura_detaylari SET 
        birim_fiyat_kurus = ?,
        kdv_tutari_kurus = ?,
        toplam_tutar_kurus = ?,
        kdv_orani_basis = ?
        WHERE id = ?");
    
    $count = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $birimFiyatKurus = MoneyHelper::toKurus($row['birim_fiyat']);
        $kdvTutariKurus = MoneyHelper::toKurus($row['kdv_tutari']);
        $toplamTutarKurus = MoneyHelper::toKurus($row['toplam_tutar']);
        $kdvBasis = TaxEngine::percentToBasis((float)$row['kdv_orani']);
        
        $updateStmt->execute([
            $birimFiyatKurus, 
            $kdvTutariKurus, 
            $toplamTutarKurus,
            $kdvBasis,
            $row['id']
        ]);
        $count++;
    }
    echo "    → $count fatura detayı dönüştürüldü\n";
    
    // borclar
    echo "  - borclar dönüştürülüyor...\n";
    $stmt = $db->query("SELECT id, tutar, toplam_kdv, indirim_tutari, odenen_tutar FROM borclar");
    $updateStmt = $db->prepare("UPDATE borclar SET 
        tutar_kurus = ?,
        toplam_kdv_kurus = ?,
        indirim_tutari_kurus = ?,
        odenen_tutar_kurus = ?
        WHERE id = ?");
    
    $count = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $tutarKurus = MoneyHelper::toKurus($row['tutar']);
        $kdvKurus = MoneyHelper::toKurus($row['toplam_kdv']);
        $indirimKurus = MoneyHelper::toKurus($row['indirim_tutari']);
        $odenenKurus = MoneyHelper::toKurus($row['odenen_tutar']);
        
        $updateStmt->execute([
            $tutarKurus,
            $kdvKurus,
            $indirimKurus,
            $odenenKurus,
            $row['id']
        ]);
        $count++;
    }
    echo "    → $count borç kaydı dönüştürüldü\n";
    
    // odemeler
    echo "  - odemeler dönüştürülüyor...\n";
    $stmt = $db->query("SELECT id, tutar FROM odemeler");
    $updateStmt = $db->prepare("UPDATE odemeler SET tutar_kurus = ? WHERE id = ?");
    
    $count = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $tutarKurus = MoneyHelper::toKurus($row['tutar']);
        $updateStmt->execute([$tutarKurus, $row['id']]);
        $count++;
    }
    echo "    → $count ödeme kaydı dönüştürüldü\n";
    
    // stok_hareketleri
    echo "  - stok_hareketleri dönüştürülüyor...\n";
    $stmt = $db->query("SELECT id, birim_fiyat FROM stok_hareketleri WHERE birim_fiyat IS NOT NULL");
    $updateStmt = $db->prepare("UPDATE stok_hareketleri SET birim_fiyat_kurus = ? WHERE id = ?");
    
    $count = 0;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $fiyatKurus = MoneyHelper::toKurus($row['birim_fiyat']);
        $updateStmt->execute([$fiyatKurus, $row['id']]);
        $count++;
    }
    echo "    → $count stok hareketi dönüştürüldü\n";
    
    echo "✓ Veri dönüşümü tamamlandı\n\n";
    
    // =====================================================
    // PHASE 3: KDV AYARI EKLE
    // =====================================================
    echo "PHASE 3: Sistem ayarları güncelleniyor...\n";
    
    // Default KDV hesaplama tipi ekle
    $checkStmt = $db->prepare("SELECT COUNT(*) FROM ayarlar WHERE ayar_adi = 'kdv_hesaplama_tipi'");
    $checkStmt->execute();
    
    if ($checkStmt->fetchColumn() == 0) {
        $insertStmt = $db->prepare("INSERT INTO ayarlar (sirket_id, ayar_adi, ayar_degeri, aciklama) 
            VALUES (1, 'kdv_hesaplama_tipi', 'kdv_haric', 'KDV Hesaplama Modu: kdv_haric veya kdv_dahil')");
        $insertStmt->execute();
        echo "  ✓ Default KDV ayarı eklendi (kdv_haric)\n";
    } else {
        echo "  ℹ KDV ayarı zaten mevcut\n";
    }
    
    echo "✓ Sistem ayarları güncellendi\n\n";
    
    // =====================================================
    // COMMIT
    // =====================================================
    $db->commit();
    
    echo "==========================================================\n";
    echo "  ✓ MİGRATION BAŞARIYLA TAMAMLANDI!\n";
    echo "==========================================================\n\n";
    
    echo "Sonraki adımlar:\n";
    echo "1. Test scriptlerini çalıştırın: php tests/TaxEngineTest.php\n";
    echo "2. Bir test faturası oluşturun\n";
    echo "3. Hesaplamaları kontrol edin\n";
    echo "4. Eski decimal sütunları kaldırmayı düşünebilirsiniz (opsiyonel)\n\n";
    
} catch (Exception $e) {
    $db->rollBack();
    
    echo "\n==========================================================\n";
    echo "  ✗ MİGRATION BAŞARISIZ!\n";
    echo "==========================================================\n\n";
    echo "Hata: " . $e->getMessage() . "\n";
    echo "Dosya: " . $e->getFile() . "\n";
    echo "Satır: " . $e->getLine() . "\n\n";
    echo "Tüm değişiklikler geri alındı (rollback).\n";
    
    exit(1);
}
