<?php
// update.php

session_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set language from GET parameter, cookie, or default to Turkish
$lang = $_GET['lang'] ?? $_COOKIE['lang'] ?? 'tr';

// Save language preference in a cookie for 30 days and in session
if (isset($_GET['lang'])) {
    $lang = $_GET['lang'];
    setcookie('lang', $lang, time() + (86400 * 30), '/'); // 30 days
    $_SESSION['lang'] = $lang; // Also store in session for immediate use
} elseif (isset($_SESSION['lang'])) {
    $lang = $_SESSION['lang'];
} elseif (isset($_COOKIE['lang'])) {
    $lang = $_COOKIE['lang'];
}

// Ensure we have a valid language
if (!in_array($lang, ['tr', 'en'])) {
    $lang = 'tr';
}

// Initialize translations array
$translations = [];

// Load language file
$lang_file = __DIR__ . '/languages/' . $lang . '.php';
if (file_exists($lang_file)) {
    $translations = include $lang_file;
    if (!is_array($translations)) {
        $translations = [];
    }
} else {
    // Fallback to Turkish if language file doesn't exist
    $lang = 'tr';
    $translations = include __DIR__ . '/languages/tr.php';
    if (!is_array($translations)) {
        $translations = [];
    }
}

// Make sure the language is set in session
$_SESSION['lang'] = $lang;

// Translation function if not exists
if (!function_exists('__')) {
    function __($key, $params = []) {
        global $translations;
        $translation = $translations[$key] ?? $key;
        
        // Replace parameters if any
        if (!empty($params) && is_array($params)) {
            foreach ($params as $param => $value) {
                $translation = str_replace(':' . $param, $value, $translation);
            }
        }
        
        return $translation;
    }
}

function run_query($db, $sql, $params = []) {
    try {
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        // Hata durumunda işlemi durdur ve hatayı göster
        throw new Exception("SQL Hatası: " . $e->getMessage() . "<br>Sorgu: " . $sql);
    }
}

function get_db_schema($db) {
    $schema = [];
    $tables = run_query($db, "SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);

    foreach ($tables as $table) {
        $schema[$table] = [
            'columns' => [],
            'indexes' => [],
            'foreign_keys' => [],
        ];

        // Sütunları al
        $columns = run_query($db, "SHOW COLUMNS FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($columns as $col) {
            $schema[$table]['columns'][$col['Field']] = [
                'Type' => $col['Type'],
                'Null' => $col['Null'],
                'Key' => $col['Key'],
                'Default' => $col['Default'],
                'Extra' => $col['Extra'],
            ];
        }

        // İndeksleri al
        $indexes = run_query($db, "SHOW INDEX FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($indexes as $idx) {
            $key_name = $idx['Key_name'];
            if (!isset($schema[$table]['indexes'][$key_name])) {
                $schema[$table]['indexes'][$key_name] = [
                    'columns' => [],
                    'unique' => ($idx['Non_unique'] == 0),
                ];
            }
            $schema[$table]['indexes'][$key_name]['columns'][] = $idx['Column_name'];
        }

        // Foreign Key'leri al
        $fks = run_query($db, "SELECT CONSTRAINT_NAME, COLUMN_NAME, REFERENCED_TABLE_NAME, REFERENCED_COLUMN_NAME
                                 FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
                                 WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND REFERENCED_TABLE_NAME IS NOT NULL", [$table])->fetchAll(PDO::FETCH_ASSOC);
        foreach ($fks as $fk) {
            $schema[$table]['foreign_keys'][$fk['CONSTRAINT_NAME']] = [
                'column' => $fk['COLUMN_NAME'],
                'referenced_table' => $fk['REFERENCED_TABLE_NAME'],
                'referenced_column' => $fk['REFERENCED_COLUMN_NAME'],
            ];
        }
    }
    return $schema;
}

function parse_sql_schema($sql_content) {
    $schema = [];
    $commands = explode(';', $sql_content);

    foreach ($commands as $command) {
        $command = trim($command);
        if (empty($command)) continue;

        // CREATE TABLE
        if (preg_match('/CREATE TABLE `(.*?)` \((.*)\)/is', $command, $matches)) {
            $table_name = $matches[1];
            $column_definitions = $matches[2];
            $schema[$table_name] = [
                'columns' => [],
                'indexes' => [],
                'foreign_keys' => [],
            ];

            // Sütunları parse et
            preg_match_all('/`([^`]+)`\s+((?:ENUM\s*\(.*?\)|(?:VAR)?CHAR\s*\(.*?\)|[a-z_]+)(?:\s+(?:NOT\s+)?NULL)?)(?:\s+DEFAULT\s+(.*?))?(?:\s+COLLATE\s+[^\s]+)?(?:\s+(AUTO_INCREMENT|PRIMARY KEY|UNIQUE KEY|KEY|CONSTRAINT.*?(?:FOREIGN KEY.*?REFERENCES.*?)\(.*?\))(?![^()]*\)))?/is', $column_definitions, $col_matches, PREG_SET_ORDER);

            foreach ($col_matches as $col_match) {
                $col_name = $col_match[1];
                $col_type_null_key = $col_match[2];
                $col_default = isset($col_match[3]) ? $col_match[3] : null;
                $col_extra = isset($col_match[4]) ? $col_match[4] : null;

                if (strpos($col_extra, 'PRIMARY KEY') !== false) {
                    $schema[$table_name]['indexes']['PRIMARY'] = ['columns' => [$col_name], 'unique' => true];
                } elseif (strpos($col_extra, 'UNIQUE KEY') !== false) {
                    preg_match('/UNIQUE KEY `(.*?)` \(`(.*?)`\)/', $col_extra, $unique_matches);
                    if (isset($unique_matches[1]) && isset($unique_matches[2])) {
                         $schema[$table_name]['indexes'][$unique_matches[1]] = ['columns' => explode('`,`', $unique_matches[2]), 'unique' => true];
                    }
                } elseif (strpos($col_extra, 'KEY') !== false) {
                    preg_match('/KEY `(.*?)` \(`(.*?)`\)/', $col_extra, $key_matches);
                    if (isset($key_matches[1]) && isset($key_matches[2])) {
                         $schema[$table_name]['indexes'][$key_matches[1]] = ['columns' => explode('`,`', $key_matches[2]), 'unique' => false];
                    }
                } elseif (strpos($col_extra, 'CONSTRAINT') !== false) {
                    preg_match('/CONSTRAINT `(.*?)` FOREIGN KEY \(`(.*?)`\) REFERENCES `(.*?)` \(`(.*?)`\)/', $col_extra, $fk_matches);
                    if (isset($fk_matches[1]) && isset($fk_matches[2]) && isset($fk_matches[3]) && isset($fk_matches[4])) {
                         $schema[$table_name]['foreign_keys'][$fk_matches[1]] = [
                            'column' => $fk_matches[2],
                            'referenced_table' => $fk_matches[3],
                            'referenced_column' => $fk_matches[4],
                        ];
                    }
                }
                
                if (!empty($col_name) && !isset($schema[$table_name]['columns'][$col_name])) {
                    $parsed_type = $col_type_null_key; // Start with the full type/null string
                    $parsed_null = 'YES'; // Assume nullable by default

                    // Check for NOT NULL and remove it from type string
                    if (preg_match('/\\s+NOT\\s+NULL$/i', $parsed_type)) {
                        $parsed_null = 'NO';
                        $parsed_type = preg_replace('/\\s+NOT\\s+NULL$/i', '', $parsed_type);
                    }
                    // Check for NULL (explicitly nullable)
                    elseif (preg_match('/\\s+NULL$/i', $parsed_type)) {
                        $parsed_null = 'YES'; // Already default, but explicit
                        $parsed_type = preg_replace('/\\s+NULL$/i', '', $parsed_type);
                    }
                    // Trim any trailing whitespace from the type
                    $parsed_type = trim($parsed_type);

                    // Correct default value handling:
                    $final_default = null;
                    if ($col_default !== null) {
                        if (strtolower($col_default) === 'null') {
                            $final_default = null; // Store as actual NULL for consistency with PDO
                        } elseif (strtolower($col_default) === 'current_timestamp()' || strtolower($col_default) === 'current_timestamp') {
                            $final_default = 'CURRENT_TIMESTAMP'; // Standardize for comparison
                        } else {
                            $final_default = trim($col_default, "'"); // Remove quotes from string defaults
                        }
                    }

                    $schema[$table_name]['columns'][$col_name] = [
                        'Type' => $parsed_type,
                        'Null' => $parsed_null,
                        'Key' => (strpos($col_extra, 'PRIMARY KEY') !== false) ? 'PRI' : ((strpos($col_extra, 'UNIQUE KEY') !== false) ? 'UNI' : ((strpos($col_extra, 'KEY') !== false) ? 'MUL' : '')),
                        'Default' => $final_default,
                        'Extra' => (strpos($col_extra, 'AUTO_INCREMENT') !== false) ? 'auto_increment' : '',
                    ];
                }
            }
        }
        // ALTER TABLE (ADD COLUMN, ADD INDEX, ADD CONSTRAINT)
        elseif (preg_match('/ALTER TABLE `(.*?)`\s+ADD COLUMN `([^`]+)`\s+(.*?)(?:;|$)/is', $command, $matches)) {
            $table_name = $matches[1];
            $col_name = $matches[2];
            $full_column_definition = trim($matches[3]);
            
            // Parse the full column definition
            $col_default = null;
            $col_type_null = $full_column_definition;
            
            // Extract DEFAULT value if present
            if (preg_match('/(.+?)\s+DEFAULT\s+(.*?)(?:\s+(?:COLLATE|AFTER|FIRST|$))/is', $full_column_definition, $default_matches)) {
                $col_type_null = trim($default_matches[1]);
                $col_default = trim($default_matches[2]);
            } elseif (preg_match('/(.+?)\s+DEFAULT\s+(.*)$/is', $full_column_definition, $default_matches)) {
                $col_type_null = trim($default_matches[1]);
                $col_default = trim($default_matches[2]);
            }

            if (!isset($schema[$table_name])) {
                $schema[$table_name] = ['columns' => [], 'indexes' => [], 'foreign_keys' => []];
            }
            $parsed_type = $col_type_null; // Start with the full type/null string
            $parsed_null = 'YES'; // Assume nullable by default

            // Check for NOT NULL and remove it from type string
            if (preg_match('/\\s+NOT\\s+NULL$/i', $parsed_type)) {
                $parsed_null = 'NO';
                $parsed_type = preg_replace('/\\s+NOT\\s+NULL$/i', '', $parsed_type);
            }
            // Check for NULL (explicitly nullable)
            elseif (preg_match('/\\s+NULL$/i', $parsed_type)) {
                $parsed_null = 'YES'; // Already default, but explicit
                $parsed_type = preg_replace('/\\s+NULL$/i', '', $parsed_type);
            }
            // Trim any trailing whitespace from the type
            $parsed_type = trim($parsed_type);

            // Correct default value handling:
            $final_default = null;
            if ($col_default !== null) {
                if (strtolower($col_default) === 'null') {
                    $final_default = null; // Store as actual NULL for consistency with PDO
                } elseif (strtolower($col_default) === 'current_timestamp()' || strtolower($col_default) === 'current_timestamp') {
                    $final_default = 'CURRENT_TIMESTAMP'; // Standardize for comparison
                } else {
                    $final_default = trim($col_default, "'"); // Remove quotes from string defaults
                }
            }

            $schema[$table_name]['columns'][$col_name] = [
                'Type' => $parsed_type,
                'Null' => $parsed_null,
                'Key' => '',
                'Default' => $final_default,
                'Extra' => '',
            ];
        } elseif (preg_match('/ALTER TABLE `(.*?)`\s+ADD (UNIQUE )?KEY `(.*?)` \(`(.*?)`\)/is', $command, $matches)) {
            $table_name = $matches[1];
            $unique = !empty($matches[2]);
            $index_name = $matches[3];
            $columns = explode('`,`', $matches[4]);
            if (!isset($schema[$table_name])) {
                $schema[$table_name] = ['columns' => [], 'indexes' => [], 'foreign_keys' => []];
            }
            $schema[$table_name]['indexes'][$index_name] = ['columns' => $columns, 'unique' => $unique];
        } elseif (preg_match('/ALTER TABLE `(.*?)`\s+ADD CONSTRAINT `(.*?)` FOREIGN KEY \(`(.*?)`\) REFERENCES `(.*?)` \(`(.*?)`\)(?: ON DELETE (.*))?(?: ON UPDATE (.*))?/is', $command, $matches)) {
            $table_name = $matches[1];
            $constraint_name = $matches[2];
            $column_name = $matches[3];
            $ref_table = $matches[4];
            $ref_column = $matches[5];
            
            if (!isset($schema[$table_name])) {
                $schema[$table_name] = ['columns' => [], 'indexes' => [], 'foreign_keys' => []];
            }
            $schema[$table_name]['foreign_keys'][$constraint_name] = [
                'column' => $column_name,
                'referenced_table' => $ref_table,
                'referenced_column' => $ref_column,
            ];
        }
    }
    return $schema;
}

function compare_schemas($current_schema, $target_schema) {
    $differences = [
        'missing_tables' => [],
        'missing_columns' => [],
        'missing_indexes' => [],
        'missing_foreign_keys' => [],
        'column_differences' => [], // Bu satırı koruyalım ama doldurmayalım
    ];

    // Eksik tabloları bul
    foreach ($target_schema as $table_name => $table_def) {
        if (!isset($current_schema[$table_name])) {
            $differences['missing_tables'][] = $table_name;
        } else {
            // Eksik sütunları bul
            foreach ($table_def['columns'] as $col_name => $col_def) {
                if (!isset($current_schema[$table_name]['columns'][$col_name])) {
                    $differences['missing_columns'][] = "{$table_name}.{$col_name}";
                } else {
                    // Sütun özellik farklılıkları - BU KISMI DEVRE DIŞI BIRAKIYORUZ
                    /*
                    foreach ($col_def as $key => $value) {
                        // NULL ve DEFAULT değerlerinin karşılaştırılması özel dikkat gerektirir.
                        // SQL parse işleminden gelen varsayılan değerler string olabilirken,
                        // veritabanından gelenler NULL veya farklı formatlarda olabilir.
                        if ($key === 'Default') {
                            $current_default = ($current_schema[$table_name]['columns'][$col_name]['Default'] === null) ? 'NULL' : $current_schema[$table_name]['columns'][$col_name]['Default'];
                            $target_default = ($value === null) ? 'NULL' : $value;
                            if (strtolower($current_default) !== strtolower($target_default)) {
                                $differences['column_differences'][] = "{$table_name}.{$col_name} -> {$key}: Mevcut='{$current_default}', Hedef='{$target_default}'";
                            }
                        } else if ($key === 'Type') {
                            // 'int(11)' ile 'int(10) unsigned' gibi durumları ele almak için basitleştirme
                            $current_type = preg_replace('/\([^)]+\)/', '', $current_schema[$table_name]['columns'][$col_name]['Type']);
                            $target_type = preg_replace('/\([^)]+\)/', '', $value);
                            if (strtolower($current_type) !== strtolower($target_type)) {
                                $differences['column_differences'][] = "{$table_name}.{$col_name} -> {$key}: Mevcut='{$current_schema[$table_name]['columns'][$col_name]['Type']}', Hedef='{$value}'";
                            }
                        } else {
                            if (isset($current_schema[$table_name]['columns'][$col_name][$key]) && strtolower($current_schema[$table_name]['columns'][$col_name][$key]) !== strtolower($value)) {
                                $differences['column_differences'][] = "{$table_name}.{$col_name} -> {$key}: Mevcut='{$current_schema[$table_name]['columns'][$col_name][$key]}', Hedef='{$value}'";
                            }
                        }
                    }
                    */
                }
            }

            // Eksik indeksleri bul
            foreach ($table_def['indexes'] as $idx_name => $idx_def) {
                // PRIMARY KEY indeks adları bazen farklı dönebilir (örn. 'PRIMARY')
                $found = false;
                foreach ($current_schema[$table_name]['indexes'] as $curr_idx_name => $curr_idx_def) {
                    if (count(array_diff($idx_def['columns'], $curr_idx_def['columns'])) === 0 && count(array_diff($curr_idx_def['columns'], $idx_def['columns'])) === 0 && $idx_def['unique'] === $curr_idx_def['unique']) {
                        $found = true;
                        break;
                    }
                }
                if (!$found) {
                    $differences['missing_indexes'][] = "{$table_name}.{$idx_name} (Sütunlar: " . implode(', ', $idx_def['columns']) . ")";
                }
            }

            // Eksik Foreign Key'leri bul
            foreach ($table_def['foreign_keys'] as $fk_name => $fk_def) {
                if (!isset($current_schema[$table_name]['foreign_keys'][$fk_name])) {
                    $found_fk = false;
                    foreach ($current_schema[$table_name]['foreign_keys'] as $curr_fk_name => $curr_fk_def) {
                        if ($curr_fk_def['column'] === $fk_def['column'] && 
                            $curr_fk_def['referenced_table'] === $fk_def['referenced_table'] && 
                            $curr_fk_def['referenced_column'] === $fk_def['referenced_column']) {
                            $found_fk = true;
                            break;
                        }
                    }
                    if (!$found_fk) {
                        $differences['missing_foreign_keys'][] = "{$table_name}.{$fk_name} (Sütun: {$fk_def['column']}, Referans: {$fk_def['referenced_table']}.{$fk_def['referenced_column']})";
                    }
                }
            }
        }
    }

    return $differences;
}

$message = '';
$success = false;
$schema_differences = [];
$display_form = true;
$sql_to_execute = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['confirm_update']) && $_POST['confirm_update'] === 'yes') {
        // Kullanıcı onayı alındı, SQL komutlarını çalıştır
        if (isset($_SESSION['db_config']) && isset($_SESSION['sql_to_execute'])) {
            $db_config = $_SESSION['db_config'];
            $host = $db_config['host'];
            $dbname = $db_config['dbname'];
            $username = $db_config['username'];
            $password = $db_config['password'];
            $sql_commands_to_execute = $_SESSION['sql_to_execute'];

            try {
                $db = new PDO("mysql:host={$db_config['host']};dbname={$db_config['dbname']};charset=utf8mb4", 
                             $db_config['username'], $db_config['password']);
                $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                
                // Debug: SQL komutlarını listele
                echo '<div class="alert alert-info">';
                echo '<h5>Çalıştırılacak SQL Komutları (' . count($_SESSION['sql_to_execute']) . ' adet):</h5>';
                echo '<pre style="max-height: 300px; overflow-y: auto;">';
                foreach ($_SESSION['sql_to_execute'] as $index => $sql) {
                    echo ($index + 1) . ". " . htmlspecialchars($sql) . "\n\n";
                }
                echo '</pre>';
                echo '</div>';
                
                $success_count = 0;

                foreach ($_SESSION['sql_to_execute'] as $sql_command) {
                    // Yabancı anahtar komutlarını şimdilik atla
                    if (strpos($sql_command, 'ADD CONSTRAINT') !== false && strpos($sql_command, 'FOREIGN KEY') !== false) {
                        echo '<div class="alert alert-info">Atlandı (FK): ' . htmlspecialchars(substr($sql_command, 0, 50)) . '...</div>';
                        continue;
                    }
                    
                    // Yabancı anahtar komutlarını çalıştırmadan önce kontrol et (eski kod - artık kullanılmıyor)
                    if (false && strpos($sql_command, 'ADD CONSTRAINT') !== false && strpos($sql_command, 'FOREIGN KEY') !== false) {
                        // Yabancı anahtar komutundan tablo ve sütun bilgilerini çıkar
                        if (preg_match('/ALTER TABLE `([^`]+)`.*?FOREIGN KEY \(`([^`]+)`\) REFERENCES `([^`]+)` \(`([^`]+)`\)/', $sql_command, $matches)) {
                            $child_table = $matches[1];
                            $child_column = $matches[2];
                            $parent_table = $matches[3];
                            $parent_column = $matches[4];
                            
                            // Tabloların varlığını kontrol et
                            $parent_exists = $db->query("SHOW TABLES LIKE '{$parent_table}'")->rowCount() > 0;
                            $child_exists = $db->query("SHOW TABLES LIKE '{$child_table}'")->rowCount() > 0;
                            
                            if (!$parent_exists) {
                                echo '<div class="alert alert-warning">Uyarı: ' . $parent_table . ' tablosu bulunamadı. FK atlanıyor: ' . htmlspecialchars($sql_command) . '</div>';
                                continue;
                            }
                            
                            if (!$child_exists) {
                                echo '<div class="alert alert-warning">Uyarı: ' . $child_table . ' tablosu bulunamadı. FK atlanıyor: ' . htmlspecialchars($sql_command) . '</div>';
                                continue;
                            }
                            
                            // Sütunların varlığını kontrol et
                            $child_column_exists = $db->query("SHOW COLUMNS FROM `{$child_table}` LIKE '{$child_column}'")->rowCount() > 0;
                            $parent_column_exists = $db->query("SHOW COLUMNS FROM `{$parent_table}` LIKE '{$parent_column}'")->rowCount() > 0;
                            
                            if (!$child_column_exists) {
                                echo '<div class="alert alert-warning">Uyarı: ' . $child_table . '.' . $child_column . ' sütunu bulunamadı. FK atlanıyor: ' . htmlspecialchars($sql_command) . '</div>';
                                continue;
                            }
                            
                            if (!$parent_column_exists) {
                                echo '<div class="alert alert-warning">Uyarı: ' . $parent_table . '.' . $parent_column . ' sütunu bulunamadı. FK atlanıyor: ' . htmlspecialchars($sql_command) . '</div>';
                                continue;
                            }
                        }
                    }
                    
                                         try {
                         $db->exec($sql_command);
                         $success_count++;
                         echo '<div class="alert alert-success">'.__('success').': ' . htmlspecialchars(substr($sql_command, 0, 50)) . '...</div>';
                     } catch (Exception $e) {
                         echo '<div class="alert alert-danger">'.__('error').': ' . htmlspecialchars(substr($sql_command, 0, 50)) . '... - ' . $e->getMessage() . '</div>';
                     }
                }

                // Başarılı güncelleme sonrası v1'den v2'ye geçiş işlemleri
                try {
                    echo '<div class="alert alert-info"><h5>'.__('v1_to_v2_migration').':</h5>';
                    
                    // 1. ID=0 olan şirket kayıtlarını düzelt
                    $stmt = $db->query("SELECT COUNT(*) as count FROM sirketler WHERE id = 0");
                    $sirket_sifir_sayisi = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                    
                    if ($sirket_sifir_sayisi > 0) {
                        // ID=0 olan şirketleri ID=1'e güncelle
                        $db->exec("UPDATE sirketler SET id = 1 WHERE id = 0");
                        echo '<div class="alert alert-success">✓ '.__('company_records_updated').'</div>';
                    }
                    
                    // 1.5. Bozuk şirket kayıtlarını temizle
                    $db->exec("DELETE FROM sirketler WHERE id = 0");
                    $db->exec("DELETE FROM kullanici_sirket_iliskisi WHERE sirket_id = 0 OR id = 0");
                    
                    // 2. Varsayılan şirket oluştur (eğer yoksa)
                    $stmt = $db->query("SELECT COUNT(*) as count FROM sirketler WHERE id = 1");
                    $sirket_sayisi = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                    
                    if ($sirket_sayisi == 0) {
                        // Varsayılan şirket oluştur
                        $varsayilan_sirket_sql = "INSERT INTO sirketler (id, sirket_adi, para_birimi, aktif_mi, teklifler_modulu_aktif, stok_islemleri_aktif, coklu_para_birimi) 
                                                  VALUES (1, 'Varsayılan Şirket', 'TRY', 1, 1, 0, 0)";
                        $db->exec($varsayilan_sirket_sql);
                        echo '<div class="alert alert-success">✓ Varsayılan şirket oluşturuldu</div>';
                    } else {
                        echo '<div class="alert alert-info">✓ Varsayılan şirket zaten mevcut</div>';
                    }
                    
                    // 3. Mevcut kullanıcıları varsayılan şirketle ilişkilendir
                    $stmt = $db->query("SELECT id, kullanici_adi, ad_soyad FROM kullanicilar WHERE aktif = 1");
                    $kullanicilar = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $iliskilendirilen_kullanici = 0;
                    foreach ($kullanicilar as $kullanici) {
                        // Kullanıcının zaten bu şirketle ilişkisi var mı kontrol et
                        $stmt = $db->prepare("SELECT COUNT(*) as count FROM kullanici_sirket_iliskisi WHERE kullanici_id = ? AND sirket_id = 1");
                        $stmt->execute([$kullanici['id']]);
                        $iliskisi_var = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                        
                        if ($iliskisi_var == 0) {
                            // İlişki yoksa ekle
                            $stmt = $db->prepare("INSERT INTO kullanici_sirket_iliskisi (kullanici_id, sirket_id, rol) VALUES (?, 1, 'admin')");
                            $stmt->execute([$kullanici['id']]);
                            $iliskilendirilen_kullanici++;
                            echo '<div class="alert alert-success">✓ Kullanıcı ilişkilendirildi: ' . htmlspecialchars($kullanici['ad_soyad']) . ' (' . htmlspecialchars($kullanici['kullanici_adi']) . ')</div>';
                        } else {
                            echo '<div class="alert alert-info">✓ Kullanıcı zaten ilişkili: ' . htmlspecialchars($kullanici['ad_soyad']) . ' (' . htmlspecialchars($kullanici['kullanici_adi']) . ')</div>';
                        }
                    }
                    
                    // Özellikle ID=1 olan kullanıcıyı kontrol et ve ilişkilendir
                    $stmt = $db->prepare("SELECT COUNT(*) as count FROM kullanici_sirket_iliskisi WHERE kullanici_id = 1 AND sirket_id = 1");
                    $stmt->execute();
                    $kullanici1_iliskisi = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                    
                    if ($kullanici1_iliskisi == 0) {
                        $stmt = $db->prepare("INSERT INTO kullanici_sirket_iliskisi (kullanici_id, sirket_id, rol) VALUES (1, 1, 'admin')");
                        $stmt->execute();
                        echo '<div class="alert alert-success">✓ ID=1 kullanıcısı özel olarak ilişkilendirildi</div>';
                        $iliskilendirilen_kullanici++;
                    }
                    
                    // Tüm kullanıcıların en az bir şirketle ilişkili olduğundan emin ol
                    $stmt = $db->query("SELECT k.id, k.kullanici_adi, k.ad_soyad 
                                       FROM kullanicilar k 
                                       LEFT JOIN kullanici_sirket_iliskisi ksi ON k.id = ksi.kullanici_id 
                                       WHERE k.aktif = 1 AND ksi.kullanici_id IS NULL");
                    $iliskisiz_kullanicilar = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    foreach ($iliskisiz_kullanicilar as $kullanici) {
                        $stmt = $db->prepare("INSERT INTO kullanici_sirket_iliskisi (kullanici_id, sirket_id, rol) VALUES (?, 1, 'admin')");
                        $stmt->execute([$kullanici['id']]);
                        echo '<div class="alert alert-success">✓ İlişkisiz kullanıcı ilişkilendirildi: ' . htmlspecialchars($kullanici['ad_soyad']) . ' (' . htmlspecialchars($kullanici['kullanici_adi']) . ')</div>';
                        $iliskilendirilen_kullanici++;
                    }
                    
                    // 4. Mevcut verileri varsayılan şirkete ata (sirket_id = 1)
                    $tables_to_update = [
                        'cariler', 'cari_gruplari', 'borclar', 'kategoriler', 'hesaplar', 
                        'hesap_hareketleri', 'urunler', 'stok_hareketleri', 'mevcut_stok',
                        'teklifler', 'teklif_detaylari', 'fatura_detaylari', 'hatirlaticilar',
                        'ayarlar', 'sistem_loglari', 'oturum_loglari', 'aktif_oturumlar',
                        'kullanici_sifre_gecmisi', 'silinen_kayitlar'
                    ];
                    
                    $guncellenen_tablolar = 0;
                    foreach ($tables_to_update as $table) {
                        // Tablo var mı kontrol et
                        $stmt = $db->query("SHOW TABLES LIKE '$table'");
                        if ($stmt->rowCount() > 0) {
                            // sirket_id sütunu var mı kontrol et
                            $stmt = $db->query("SHOW COLUMNS FROM `$table` LIKE 'sirket_id'");
                            if ($stmt->rowCount() > 0) {
                                // sirket_id = 0 veya NULL olan kayıtları güncelle
                                $stmt = $db->prepare("UPDATE `$table` SET sirket_id = 1 WHERE sirket_id = 0 OR sirket_id IS NULL");
                                $stmt->execute();
                                $etkilenen_satir = $stmt->rowCount();
                                if ($etkilenen_satir > 0) {
                                    echo '<div class="alert alert-success">✓ Tablo güncellendi: ' . $table . ' (' . $etkilenen_satir . ' kayıt)</div>';
                                    $guncellenen_tablolar++;
                                }
                            }
                        }
                    }
                    
                    // 5. AUTO_INCREMENT değerlerini düzelt
                    $db->exec("ALTER TABLE sirketler AUTO_INCREMENT = 2");
                    $db->exec("ALTER TABLE kullanici_sirket_iliskisi AUTO_INCREMENT = 1");
                    echo '<div class="alert alert-success">✓ AUTO_INCREMENT değerleri düzeltildi</div>';
                    
                    // 5.5. Son kontrol ve temizlik
                    // Bozuk ilişkileri temizle
                    $db->exec("DELETE FROM kullanici_sirket_iliskisi WHERE kullanici_id NOT IN (SELECT id FROM kullanicilar)");
                    $db->exec("DELETE FROM kullanici_sirket_iliskisi WHERE sirket_id NOT IN (SELECT id FROM sirketler)");
                    
                    // Şirket adını kontrol et ve düzelt
                    $stmt = $db->query("SELECT sirket_adi FROM sirketler WHERE id = 1");
                    $sirket_adi = $stmt->fetch(PDO::FETCH_COLUMN);
                    if (empty($sirket_adi) || $sirket_adi == 'NULL') {
                        $db->exec("UPDATE sirketler SET sirket_adi = 'Varsayılan Şirket' WHERE id = 1");
                        echo '<div class="alert alert-success">✓ Şirket adı düzeltildi</div>';
                    }
                    
                    // 6. Varsayılan şirket bilgilerini session'a ekle
                    $stmt = $db->query("SELECT id, sirket_adi FROM sirketler WHERE id = 1");
                    $varsayilan_sirket = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($varsayilan_sirket) {
                        $_SESSION['aktif_sirket_id'] = 1;
                        $_SESSION['aktif_sirket_adi'] = $varsayilan_sirket['sirket_adi'];
                        $_SESSION['kullanici_sirketleri'] = [$varsayilan_sirket];
                        echo '<div class="alert alert-success">✓ Varsayılan şirket session\'a eklendi</div>';
                    }
                    
                    echo '<div class="alert alert-success"><strong>V1\'den V2\'ye Geçiş Tamamlandı!</strong></div>';
                    echo '<ul>';
                    echo '<li>✓ Varsayılan şirket oluşturuldu/mevcut</li>';
                    echo '<li>✓ ' . $iliskilendirilen_kullanici . ' kullanıcı ilişkilendirildi</li>';
                    echo '<li>✓ ' . $guncellenen_tablolar . ' tablo güncellendi</li>';
                    echo '<li>✓ Tüm veriler varsayılan şirkete atandı</li>';
                    echo '<li>✓ Kullanıcı-şirket ilişkileri kuruldu</li>';
                    echo '<li>✓ AUTO_INCREMENT değerleri düzeltildi</li>';
                    echo '</ul>';
                    echo '<div class="alert alert-info"><strong>3 saniye sonra giriş sayfasına yönlendirileceksiniz...</strong></div>';
                    echo '</div>';
                    
                } catch (Exception $e) {
                    echo '<div class="alert alert-danger">V1\'den V2\'ye geçiş hatası: ' . $e->getMessage() . '</div>';
                    error_log("V1'den V2'ye geçiş hatası: " . $e->getMessage());
                }
                
                $message = "Veritabanı başarıyla güncellendi! Varsayılan şirket oluşturuldu ve kullanıcınız bu şirkete atandı.";
                $success = true;
                unset($_SESSION['db_config']); // Oturum verilerini temizle
                unset($_SESSION['sql_to_execute']);
                
                // Başarılı güncelleme sonrası login sayfasına yönlendir
                echo '<script>
                    setTimeout(function() {
                        window.location.href = "views/login.php";
                    }, 3000);
                </script>';

            } catch (Exception $e) {
                $message = __('update_error_message') . $e->getMessage();
                $success = false;
            }
        } else {
            $message = __('session_info_missing');
            $success = false;
        }
        $display_form = true; // Tekrar form gösterilebilir
    } else {
        // İlk form gönderimi, şema farklılıklarını kontrol et
    $host = $_POST['host'];
    $dbname = $_POST['dbname'];
    $username = $_POST['username'];
    $password = $_POST['password'];

    try {
        $db = new PDO("mysql:host=$host", $username, $password);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $stmt = $db->query("SHOW DATABASES LIKE '$dbname'");
        if ($stmt->rowCount() == 0) {
            throw new Exception("'$dbname' veritabanı bulunamadı. Lütfen kontrol edip tekrar deneyin.");
        }
        
        $db->exec("USE `$dbname`");
        $db->exec("SET NAMES utf8mb4");

            $current_db_schema = get_db_schema($db);

            $install_sql_path = 'install.sql';
            if (!file_exists($install_sql_path)) {
                throw new Exception("'install.sql' dosyası bulunamadı. Lütfen '{$install_sql_path}' yolunu kontrol edin.");
            }
            $target_sql_content = file_get_contents($install_sql_path);
            $target_db_schema = parse_sql_schema($target_sql_content);

            $schema_differences = compare_schemas($current_db_schema, $target_db_schema);

            // Eğer fark varsa kullanıcıya göster
            if (!empty($schema_differences['missing_tables']) || !empty($schema_differences['missing_columns']) || !empty($schema_differences['missing_indexes']) || !empty($schema_differences['missing_foreign_keys'])) {
                $message = __('schema_differences_found');
                $display_form = false; // Onay formunu göster
                $_SESSION['db_config'] = ['host' => $host, 'dbname' => $dbname, 'username' => $username, 'password' => $password];
                $_SESSION['sql_to_execute'] = []; // Buraya çalıştırılacak SQL komutlarını toplayacağız

                // SQL komutlarını topla
                $sql_to_execute = [];
                
                // 1. Önce eksik tabloları oluştur
                $create_table_sqls = [];
                                 foreach ($schema_differences['missing_tables'] as $table) {
                     // install.sql'den CREATE TABLE komutunu bul
                     $pattern = '/CREATE TABLE\s+`' . preg_quote($table, '/') . '`\s*\(.*?\)\s*ENGINE=.*?;/is';
                     if (preg_match($pattern, $target_sql_content, $matches)) {
                         $create_table_sqls[] = trim($matches[0]);
                     }
                 }
                
                // CREATE TABLE IF NOT EXISTS komutlarını da ekle
                $pattern = '/CREATE TABLE IF NOT EXISTS.*?;/is';
                if (preg_match_all($pattern, $target_sql_content, $matches)) {
                    foreach ($matches[0] as $create_table_sql) {
                        $create_table_sqls[] = trim($create_table_sql);
                    }
                }
                
                // 2. Sonra eksik sütunları ekle
                $add_column_sqls = [];
                $modify_column_sqls = [];
                $add_index_sqls = [];
                $add_fk_sqls = [];
                foreach ($schema_differences['missing_columns'] as $col_full_name) {
                    list($table_name, $col_name) = explode('.', $col_full_name);
                    // Hedef şemadan sütun detaylarını al
                    $col_def = $target_db_schema[$table_name]['columns'][$col_name];

                    $col_type = $col_def['Type'];
                    $col_null = (strtoupper($col_def['Null']) === 'NO') ? ' NOT NULL' : '';
                    $col_default_part = '';
                    
                    // Check if the column type is a date/time type
                    $is_datetime_type = (stripos($col_type, 'datetime') !== false || stripos($col_type, 'timestamp') !== false);
                    // Check if the column type is an enum type
                    $is_enum_type = (stripos($col_type, 'enum') !== false);

                    if ($col_def['Default'] !== null) {
                        if (strtolower($col_def['Default']) === 'null') {
                            $col_default_part = ' DEFAULT NULL';
                        } elseif (strtolower($col_def['Default']) === 'current_timestamp' || strtolower($col_def['Default']) === 'current_timestamp()') {
                            $col_default_part = ' DEFAULT CURRENT_TIMESTAMP';
                        } elseif (($is_datetime_type || $is_enum_type) && $col_def['Default'] === '') {
                            // For datetime/timestamp/enum, an empty string default is invalid. Omit it.
                            $col_default_part = ''; 
                        } else {
                            $col_default_part = " DEFAULT '" . $col_def['Default'] . "'";
                        }
                    }
                    $col_extra_auto_increment = (strpos($col_def['Extra'], 'auto_increment') !== false) ? ' AUTO_INCREMENT' : '';

                    $add_column_sqls[] = "ALTER TABLE `{$table_name}` ADD COLUMN `{$col_name}` {$col_type}{$col_null}{$col_default_part}{$col_extra_auto_increment}";
                }

                // Sütun farklılıkları için ALTER TABLE MODIFY COLUMN komutlarını topla - BU KISMI DEVRE DIŞI BIRAKIYORUZ
                /*
                foreach ($schema_differences['column_differences'] as $col_diff_string) {
                    // Sadece tip ve nullability farklarını yakala
                    preg_match('/^(.*?)\.(.*?) -> Type: Mevcut=\'(.*?)\', Hedef=\'(.*?)\'/', $col_diff_string, $type_matches);
                    preg_match('/^(.*?)\.(.*?) -> Null: Mevcut=\'(.*?)\', Hedef=\'(.*?)\'/', $col_diff_string, $null_matches);
                    preg_match('/^(.*?)\.(.*?) -> Default: Mevcut=\'(.*?)\', Hedef=\'(.*?)\'/', $col_diff_string, $default_matches);

                    $table_name = $type_matches[1] ?? ($null_matches[1] ?? ($default_matches[1] ?? null));
                    $column_name = $type_matches[2] ?? ($null_matches[2] ?? ($default_matches[2] ?? null));

                    if ($table_name && $column_name) {
                        // Hedef şemadan sütun detaylarını al
                        $col_def = $target_db_schema[$table_name]['columns'][$column_name];

                        $col_type = $col_def['Type'];
                        $col_null = (strtoupper($col_def['Null']) === 'NO') ? ' NOT NULL' : '';
                        $col_default_part = '';
                        
                        // Check if the column type is a date/time type
                        $is_datetime_type_modify = (stripos($col_type, 'datetime') !== false || stripos($col_type, 'timestamp') !== false);
                        // Check if the column type is an enum type
                        $is_enum_type_modify = (stripos($col_type, 'enum') !== false);

                        if ($col_def['Default'] !== null) {
                            if (strtolower($col_def['Default']) === 'null') {
                                $col_default_part = ' DEFAULT NULL';
                            } elseif (strtolower($col_def['Default']) === 'current_timestamp' || strtolower($col_def['Default']) === 'current_timestamp()') {
                                $col_default_part = ' DEFAULT CURRENT_TIMESTAMP';
                            } elseif (($is_datetime_type_modify || $is_enum_type_modify) && $col_def['Default'] === '') {
                                // For datetime/timestamp/enum, an empty string default is invalid. Omit it.
                                $col_default_part = '';
                            } else {
                                $col_default_part = " DEFAULT '" . $col_def['Default'] . "'";
                            }
                        }
                        $col_extra_auto_increment = (strpos($col_def['Extra'], 'auto_increment') !== false) ? ' AUTO_INCREMENT' : '';

                        // MODIFY COLUMN komutunu oluştur
                        $modify_column_sqls[] = "ALTER TABLE `{$table_name}` MODIFY COLUMN `{$column_name}` {$col_type}{$col_null}{$col_default_part}{$col_extra_auto_increment}";
                    }
                }
                */

                // Eksik indeksler için ALTER TABLE ADD KEY/UNIQUE KEY komutlarını topla
                foreach ($schema_differences['missing_indexes'] as $index_full_name) {
                    preg_match('/^(.*?)\.(.*?)(?: \(Sütunlar: .*?\))?$/', $index_full_name, $index_parts);
                    $table_name = $index_parts[1];
                    $index_name = $index_parts[2];
                    
                    // Hedef şemadan indeks detaylarını al
                    $idx_def = $target_db_schema[$table_name]['indexes'][$index_name];
                    $columns_list = '`' . implode('`,`', $idx_def['columns']) . '`';
                    $unique_keyword = $idx_def['unique'] ? 'UNIQUE ' : '';

                    $add_index_sqls[] = "ALTER TABLE `{$table_name}` ADD {$unique_keyword}KEY `{$index_name}` ({$columns_list})";
                }

                // Eksik Foreign Key'ler için ALTER TABLE ADD CONSTRAINT komutlarını topla
                foreach ($schema_differences['missing_foreign_keys'] as $fk_full_name) {
                    preg_match('/^(.*?)\.(.*?)(?: \(Sütun: .*?, Referans: .*?\))?$/', $fk_full_name, $fk_parts);
                    $table_name = $fk_parts[1];
                    $constraint_name = $fk_parts[2];

                    // Hedef şemadan foreign key detaylarını al
                    $fk_def = $target_db_schema[$table_name]['foreign_keys'][$constraint_name];

                    $column_name = $fk_def['column'];
                    $referenced_table = $fk_def['referenced_table'];
                    $referenced_column = $fk_def['referenced_column'];

                    // install.sql'den ON DELETE/ON UPDATE aksiyonlarını çekmek için regex kullan
                    preg_match('/ALTER TABLE `'.preg_quote($table_name).'` ADD CONSTRAINT `'.preg_quote($constraint_name).'` FOREIGN KEY \(`'.preg_quote($column_name).'`\) REFERENCES `'.preg_quote($referenced_table).'` \(`'.preg_quote($referenced_column).'`\)(?:\s+ON DELETE (.*?))?(?:\s+ON UPDATE (.*?))?/is', $target_sql_content, $fk_action_matches);
                    
                    $on_delete_action = isset($fk_action_matches[1]) ? " ON DELETE " . $fk_action_matches[1] : '';
                    $on_update_action = isset($fk_action_matches[2]) ? " ON UPDATE " . $fk_action_matches[2] : '';

                    $add_fk_sqls[] = "ALTER TABLE `{$table_name}` ADD CONSTRAINT `{$constraint_name}` FOREIGN KEY (`{$column_name}`) REFERENCES `{$referenced_table}` (`{$referenced_column}`) {$on_delete_action}{$on_update_action}";
                }

                // SQL komutlarını doğru sırada birleştir (Foreign keyler şimdilik atlanıyor)
                $all_sql_commands = array_merge(
                    $create_table_sqls,      // 1. Önce tablolar
                    $add_column_sqls,        // 2. Sonra sütunlar
                    $modify_column_sqls,     // 3. Sonra sütun değişiklikleri
                    $add_index_sqls         // 4. Sonra indeksler
                    // $add_fk_sqls            // 5. Foreign keyler şimdilik atlanıyor
                );
                 
                 $_SESSION['sql_to_execute'] = $all_sql_commands;

            } else {
                $message = __('update_success_message');
                $success = true;
            }

        } catch (Exception $e) {
            $message = __('error_occurred') . ': ' . $e->getMessage();
            $success = false;
        }
    }
}

?>
<!DOCTYPE html>
<html lang="<?= $lang ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo - Veritabanı Güncelleme</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        }
        .main-container {
            max-width: 800px;
            margin: 2rem auto;
            padding: 2rem;
        }
        .hero-section {
            text-align: center;
            margin-bottom: 2rem;
            color: white;
        }
        .hero-section h1 {
            font-weight: 700;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        .card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            border: none;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            backdrop-filter: blur(10px);
        }
        .card-header {
            background: linear-gradient(135deg, #007bff, #6610f2);
            color: white;
            padding: 1.5rem;
            text-align: center;
            border-bottom: none;
        }
        .card-header h4 {
            font-weight: 700;
            margin: 0;
        }
        .card-body {
            padding: 2.5rem;
        }
        .form-label {
            font-weight: 600;
        }
        .form-control, .btn {
            border-radius: 10px;
            padding: 12px 15px;
        }
        .btn-primary {
             background: linear-gradient(135deg, #007bff, #6610f2);
             border:none;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
        }
        .btn-success {
            background: linear-gradient(135deg, #28a745, #20c997);
            border: none;
        }
    </style>
</head>
<body>
    <div class="main-container">
        <div class="hero-section">
            <h1><i class="bi bi-arrow-clockwise me-3"></i><?= __('update_database') ?></h1>
            <p><?= __('update_safely') ?></p>
        </div>

        <div class="card">
            <div class="card-header">
                <h4><?= __('update_wizard') ?></h4>
            </div>
            <div class="card-body">
                <?php if (!empty($message)): ?>
                    <div class="alert <?php echo $success ? 'alert-success' : 'alert-danger'; ?> alert-dismissible fade show" role="alert">
                         <?php if ($success): ?>
                            <i class="bi bi-check-circle-fill me-2"></i>
                         <?php else: ?>
                            <i class="bi bi-exclamation-triangle-fill me-2"></i>
                         <?php endif; ?>
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <?php if ($display_form): ?>
                    <h5 class="card-title mb-4">
                        <i class="bi bi-database-gear me-2"></i><?= __('database_information') ?>
                    </h5>
                    <p class="text-muted mb-4"><?= __('database_connection_info') ?></p>

                    <form action="update.php" method="post">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="host" class="form-label"><?= __('database_server') ?></label>
                                <input type="text" class="form-control" id="host" name="host" value="localhost" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="dbname" class="form-label"><?= __('database_name') ?></label>
                                <input type="text" class="form-control" id="dbname" name="dbname" required>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="username" class="form-label"><?= __('username') ?></label>
                                <input type="text" class="form-control" id="username" name="username" value="root" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="password" class="form-label"><?= __('password') ?></label>
                                <input type="password" class="form-control" id="password" name="password">
                            </div>
                        </div>
                        <div class="d-grid mt-3">
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="bi bi-search me-2"></i><?= __('check_schema_differences_btn') ?>
                            </button>
                        </div>
                    </form>
                <?php else: // Farklılıklar varsa veya onay bekleniyorsa ?>
                    <h5 class="card-title mb-4">
                        <i class="bi bi-exclamation-triangle-fill me-2"></i><?= __('schema_differences_found') ?>
                    </h5>
                    <p class="text-muted mb-4"><?= __('schema_differences_description') ?></p>

                    <ul class="list-group mb-4">
                        <?php if (!empty($schema_differences['missing_tables'])): ?>
                            <li class="list-group-item list-group-item-danger">
                                <strong><?= __('missing_tables') ?>:</strong>
                                <ul>
                                    <?php foreach ($schema_differences['missing_tables'] as $table): ?>
                                        <li><code><?php echo htmlspecialchars($table); ?></code></li>
                                    <?php endforeach; ?>
                                </ul>
                            </li>
                        <?php endif; ?>
                        <?php if (!empty($schema_differences['missing_columns'])): ?>
                            <li class="list-group-item list-group-item-warning">
                                <strong><?= __('missing_columns') ?>:</strong>
                                <ul>
                                    <?php foreach ($schema_differences['missing_columns'] as $column): ?>
                                        <li><code><?php echo htmlspecialchars($column); ?></code></li>
                                    <?php endforeach; ?>
                                </ul>
                            </li>
                        <?php endif; ?>
                        <?php if (!empty($schema_differences['missing_indexes'])): ?>
                            <li class="list-group-item list-group-item-warning">
                                <strong><?= __('missing_indexes') ?>:</strong>
                                <ul>
                                    <?php foreach ($schema_differences['missing_indexes'] as $index): ?>
                                        <li><code><?php echo htmlspecialchars($index); ?></code></li>
                                    <?php endforeach; ?>
                                </ul>
                            </li>
                        <?php endif; ?>
                        <?php if (!empty($schema_differences['missing_foreign_keys'])): ?>
                            <li class="list-group-item list-group-item-warning">
                                <strong><?= __('missing_foreign_keys') ?>:</strong>
                                <ul>
                                    <?php foreach ($schema_differences['missing_foreign_keys'] as $fk): ?>
                                        <li><code><?php echo htmlspecialchars($fk); ?></code></li>
                                    <?php endforeach; ?>
                                </ul>
                            </li>
                        <?php endif; ?>
                    </ul>

                    <?php if (empty($schema_differences['missing_tables']) && empty($schema_differences['missing_columns']) && empty($schema_differences['missing_indexes']) && empty($schema_differences['missing_foreign_keys']) && empty($schema_differences['column_differences'])): ?>
                    <div class="text-center">
                            <h3 class="mb-4"><i class="bi bi-check-circle-fill text-success me-2"></i><?= __('update_success') ?></h3>
                            <p class="text-muted mb-4"><?= __('no_update_needed') ?></p>
                        <div class="d-grid">
                            <a href="views/login.php" class="btn btn-success btn-lg"><i class="bi bi-box-arrow-in-right me-2"></i><?= __('login_to_system') ?></a>
                        </div>
                        <script>
                        // 3 saniye sonra otomatik yönlendirme
                        setTimeout(function() {
                            window.location.href = 'views/login.php';
                        }, 3000);
                        </script>
                    </div>
                    <?php else: ?>
                        <form action="update.php" method="post">
                            <input type="hidden" name="confirm_update" value="yes">
                            <div class="d-grid mt-4">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="bi bi-check-circle me-2"></i><?= __('confirm_and_update') ?>
                                </button>
                            </div>
                        </form>
                        <div class="d-grid mt-2">
                            <a href="update.php" class="btn btn-secondary btn-lg">
                                <i class="bi bi-x-circle me-2"></i><?= __('cancel') ?>
                            </a>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 