<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/ayar_fonksiyonlari.php';

// Bu dosya artık her sayfa açılışında kurları güncellemez.
// Son güncellemeye göre bir TTL uygular; süre dolmadıysa sessizce çıkar.

try {
    // TTL (dakika) ayarı: ayarlardan okunur, yoksa 360 dk (6 saat)
    $ttl_minutes_raw = get_site_setting($db, 'doviz_ttl_dakika', '360');
    $ttl_minutes = (int)$ttl_minutes_raw;
    if ($ttl_minutes <= 0) {
        $ttl_minutes = 360;
    }

    // Son güncelleme zamanını kontrol et
    $stmt = $db->query("SELECT MAX(guncelleme_tarihi) AS son_guncelleme FROM doviz_kurlari");
    $last_updated_at = $stmt->fetchColumn();
    if ($last_updated_at) {
        $diff_minutes = (time() - strtotime($last_updated_at)) / 60;
        if ($diff_minutes < $ttl_minutes) {
            // Güncel; işlem yapmadan çık
            return;
        }
    }

    // Basit kilit mekanizması (eşzamanlı güncellemeleri engelleme)
    $lock_file = __DIR__ . '/../ajax/doviz_update.lock';
    $lock_handle = @fopen($lock_file, 'c');
    if ($lock_handle === false) {
        // Kilit dosyası açılamazsa, güncellemeyi atla (sessiz çık)
        return;
    }
    $locked = @flock($lock_handle, LOCK_EX | LOCK_NB);
    if (!$locked) {
        // Başka bir süreç güncelliyorsa çık
        @fclose($lock_handle);
        return;
    }

    // TCMB döviz kuru XML adresi
    $url = 'https://www.tcmb.gov.tr/kurlar/today.xml';

    $response = false;
    $http_code = 0;
    $curl_err = '';

    if (function_exists('curl_init')) {
        // cURL ile ağ isteği (timeout ile)
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_TIMEOUT => 8,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_USERAGENT => 'Odjoo/1.0 (+doviz-guncelle)'
        ]);
        $response = curl_exec($ch);
        $http_code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_err = curl_error($ch);
        curl_close($ch);
    } else {
        // Fallback: file_get_contents ile (kısa timeout)
        $context = stream_context_create([
            'http' => [
                'timeout' => 8,
                'header' => "User-Agent: Odjoo/1.0 (+doviz-guncelle)\r\n"
            ],
            'ssl' => [
                'verify_peer' => true,
                'verify_peer_name' => true
            ]
        ]);
        $response = @file_get_contents($url, false, $context);
        $http_code = $response !== false ? 200 : 0;
        $curl_err = $response === false ? 'file_get_contents failed' : '';
    }

    if ($response === false || $http_code !== 200) {
        // Ağ hatası; kilidi bırak ve çık
        @flock($lock_handle, LOCK_UN);
        @fclose($lock_handle);
        @unlink($lock_file);
        error_log('Doviz guncelleme hatasi: HTTP ' . $http_code . ' - ' . $curl_err);
        return;
    }

    $xml = @simplexml_load_string($response);
    if ($xml === false) {
        @flock($lock_handle, LOCK_UN);
        @fclose($lock_handle);
        @unlink($lock_file);
        error_log('Doviz guncelleme hatasi: XML parse edilemedi');
        return;
    }

    // Gösterilecek para birimleri - TCMB'den mevcut olan tüm para birimleri
    $birimler = [
        "USD", "EUR", "GBP", "CHF", "RUB", "JPY", // Mevcut para birimleri
        "AUD", "DKK", "SEK", "CAD", "KWD", "NOK", "SAR", // Ek Avrupa ve diğer para birimleri
        "BGN", "RON", "CNY", "PKR", "QAR", "KRW", "AZN", "AED" // Asya ve Orta Doğu para birimleri
    ];

    // Dövizleri veritabanına kaydet / güncelle
    $now = date('Y-m-d H:i:s');
    $sql = "INSERT INTO doviz_kurlari (para_birimi, alis, satis, guncelleme_tarihi)
            VALUES (:code, :alis, :satis, :now)
            ON DUPLICATE KEY UPDATE
            alis = VALUES(alis),
            satis = VALUES(satis),
            guncelleme_tarihi = VALUES(guncelleme_tarihi)";
    $stmt = $db->prepare($sql);

    foreach ($xml->Currency as $currency) {
        $code = (string)$currency['CurrencyCode'];
        if (in_array($code, $birimler, true)) {
            $alis = (float)$currency->ForexBuying;
            $satis = (float)$currency->ForexSelling;
            if ($alis > 0 && $satis > 0) {
                $stmt->execute([
                    ':code' => $code,
                    ':alis' => $alis,
                    ':satis' => $satis,
                    ':now' => $now
                ]);
            }
        }
    }

    // Kilidi bırak ve temizle
    @flock($lock_handle, LOCK_UN);
    @fclose($lock_handle);
    @unlink($lock_file);
} catch (Throwable $e) {
    error_log('Doviz guncelleme istisnasi: ' . $e->getMessage());
    // Hata durumunda sessiz çık (dashboard'u engelleme)
    return;
}

?>
