<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/yetki_kontrol.php';
require_once __DIR__ . '/../includes/ayar_fonksiyonlari.php'; // get_site_setting fonksiyonu için eklendi
require_once __DIR__ . '/../functions/log_fonksiyonlari.php'; // sistem_log_ekle için eklendi
require_once __DIR__ . '/../includes/fonksiyonlar.php'; // __t fonksiyonu ve session_guvenlik fonksiyonları için eklendi

$sayfa_baslik = __t('guvenlik_kontrol_paneli');

// Sadece 'admin' yetki seviyesine sahip kullanıcıların bu sayfaya erişmesine izin ver
yetki_kontrol_ve_yonlendir('guvenlik_kontrol', 'goruntule');

$messages = []; // Genel mesajlar

// Form gönderimleri için ortak işlem bloğu
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = "Geçersiz form gönderimi!";
        header("Location: guvenlik_kontrol.php");
        exit();
    }

    // install.php dosyasını silme işlemi
    if (isset($_POST['delete_install'])) {
        $install_file = __DIR__ . '/../install.php';
        if (file_exists($install_file)) {
            if (unlink($install_file)) {
                $messages[] = ['type' => 'success', 'text' => __t('install_php_dosyasi_basariyla_silindi')];
                sistem_log_ekle($db, 'sil', 'sistem', null, null, 'install.php dosyası silindi.');
            } else {
                $messages[] = ['type' => 'danger', 'text' => __t('install_php_dosyasi_silinirken_bir_hata_olustu')];
                sistem_log_ekle($db, 'hata', 'sistem', null, null, 'install.php silinemedi.');
            }
        } else {
            $messages[] = ['type' => 'info', 'text' => __t('install_php_dosyasi_zaten_bulunmuyor')];
        }
    }

    // Session güvenlik iyileştirme işlemi
    if (isset($_POST['improve_session'])) {
        try {
            session_guvenlik_iyilestir();
            $_SESSION['mesaj'] = "Session güvenlik ayarları başarıyla iyileştirildi. Oturumunuz sonlandırılıyor...";
            sistem_log_ekle($db, 'guncelle', 'sistem', null, null, 'Oturum güvenlik ayarları iyileştirildi.');
            
            // 3 saniye sonra login sayfasına yönlendir
            header("Refresh: 3; URL=" . BASE_URL . "/views/login.php?message=session_improved");
            exit;
        } catch (Exception $e) {
            $_SESSION['hata'] = "Session güvenlik iyileştirme sırasında hata oluştu: " . $e->getMessage();
        }
    }
}

// Güvenlik skoru hesaplama için gerekli değişkenler
$security_score = 0;
$total_checks = 0;

// 1. config.php dosya izinleri kontrolü
$config_file = __DIR__ . '/../includes/config.php';
$config_perms = @fileperms($config_file);
$config_chmod = $config_perms ? substr(sprintf('%o', $config_perms), -4) : __t('bilinmiyor');
$config_status = [
    'type' => 'danger',
    'icon' => 'bi-exclamation-octagon-fill',
    'text' => __t('erisim_yok_veya_izinler_okunamıyor'),
    'suggestion' => __t('lutfen_dosya_izinlerini_kontrol_edin'),
    'score' => 0
];
$total_checks++;
if ($config_perms !== false) {
    if ($config_chmod === '0644' || $config_chmod === '0600') {
        $config_status = [
            'type' => 'success',
            'icon' => 'bi-check-circle-fill',
            'text' => __t('uygun') . ' ('.$config_chmod.')',
            'suggestion' => '',
            'score' => 20
        ];
        $security_score += 20;
    } else {
        $config_status = [
            'type' => 'warning',
            'icon' => 'bi-exclamation-triangle-fill',
            'text' => __t('onerilmiyor') . ' ('.$config_chmod.')',
            'suggestion' => __t('guvenlik_icin_0644_veya_0600_olarak_ayarlanmasi_onerilir'),
            'score' => 10
        ];
        $security_score += 10;
    }
}

// 2. HTTPS/HTTP kullanımı kontrolü
$is_https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443);
$https_status = [
    'type' => 'success',
    'icon' => 'bi-lock-fill',
    'text' => __t('https_aktif'),
    'suggestion' => __t('siteniz_https_uzerinden_guvenli_bir_sekilde_calisiyor'),
    'score' => 25
];
$total_checks++;
if (!$is_https) {
    $https_status = [
        'type' => 'danger',
        'icon' => 'bi-unlock-fill',
        'text' => __t('http_kullaniliyor'),
        'suggestion' => __t('siteniz_su_anda_http_guvenli_olmayan_baglanti_uzerinden_calisiyor'),
        'score' => 0
    ];
} else {
    $security_score += 25;
}

// 3. install.php dosyasının varlığı kontrolü
$install_file_exists = file_exists(__DIR__ . '/../install.php');
$install_status = [
    'type' => 'success',
    'icon' => 'bi-check-circle-fill',
    'text' => __t('bulunmuyor'),
    'suggestion' => __t('install_php_dosyasi_sunucunuzda_bulunmuyor_sistem_guvenli'),
    'score' => 20
];
$total_checks++;
if ($install_file_exists) {
    $install_status = [
        'type' => 'danger',
        'icon' => 'bi-exclamation-triangle-fill',
        'text' => __t('mevcut'),
        'suggestion' => __t('install_php_dosyasi_hala_sunucunuzda_bulunuyor'),
        'score' => 0
    ];
} else {
    $security_score += 20;
}

// 4. Hata günlükleri kontrolü
$error_log_file = __DIR__ . '/../includes/php_errors.log';
$error_log_status = [
    'type' => 'info',
    'icon' => 'bi-info-circle-fill',
    'text' => __t('durum_bilinmiyor'),
    'suggestion' => __t('hata_gunlugu_php_errors_log_dosyasi_bulunmuyor'),
    'score' => 5
];
$total_checks++;
if (file_exists($error_log_file)) {
    if (!is_writable($error_log_file)) {
        $error_log_status = [
            'type' => 'danger',
            'icon' => 'bi-x-circle-fill',
            'text' => __t('yazilamiyor'),
            'suggestion' => __t('hata_gunlugu_dosyasina_yazilamiyor_hatalar_kaydedilemeyebilir'),
            'score' => 0
        ];
    } else if (filesize($error_log_file) > 0) {
        $error_log_status = [
            'type' => 'warning',
            'icon' => 'bi-exclamation-triangle-fill',
            'text' => __t('hata_var') . ' (' . formatBytes(filesize($error_log_file)) . ')',
            'suggestion' => __t('hata_gunlugu_dosyasinda_kayitlar_var'),
            'score' => 10
        ];
        $security_score += 10;
    } else {
        $error_log_status = [
            'type' => 'success',
            'icon' => 'bi-check-circle-fill',
            'text' => __t('temiz'),
            'suggestion' => __t('hata_gunlugu_temiz_herhangi_bir_sorun_gorunmuyor'),
            'score' => 15
        ];
        $security_score += 15;
    }
} else {
    if (!is_writable(dirname($error_log_file))) {
        $error_log_status = [
            'type' => 'danger',
            'icon' => 'bi-exclamation-octagon-fill',
            'text' => __t('dizin_yazilamaz'),
            'suggestion' => __t('hata_gunlugu_dizini_includes_yazilabilir_degil'),
            'score' => 0
        ];
    } else {
        $security_score += 5;
    }
}

// 5. Yedekler dizini kontrolü
$backups_dir = __DIR__ . '/../views/backups';
$backups_status = [
    'type' => 'danger',
    'icon' => 'bi-x-circle-fill',
    'text' => __t('dizin_bulunamadi'),
    'suggestion' => __t('yedekler_dizini_views_backups_bulunamadi'),
    'score' => 0
];
$total_checks++;
if (is_dir($backups_dir)) {
    if (is_writable($backups_dir)) {
        $backups_status = [
            'type' => 'success',
            'icon' => 'bi-check-circle-fill',
            'text' => __t('yazilabilir'),
            'suggestion' => __t('yedekler_dizini_uygun_veritabani_yedeklemesi_yapilabilir'),
            'score' => 10
        ];
        $security_score += 10;
    } else {
        $backups_status = [
            'type' => 'danger',
            'icon' => 'bi-exclamation-triangle-fill',
            'text' => __t('yazilamiyor'),
            'suggestion' => __t('yedekler_dizini_yazilabilir_degil'),
            'score' => 0
        ];
    }
}

// 6. Session güvenliği kontrolü - Gelişmiş
$session_guvenlik = session_guvenlik_kontrol();
$session_score = $session_guvenlik['skor'];
$session_percentage = round(($session_score / ($session_guvenlik['toplam_kontrol'] * 20)) * 100); // Max skor 20 * kontrol sayısı

$session_status = [
    'type' => $session_percentage >= 80 ? 'success' : ($session_percentage >= 60 ? 'warning' : 'danger'),
    'icon' => $session_percentage >= 80 ? 'bi-shield-check' : ($session_percentage >= 60 ? 'bi-shield-exclamation' : 'bi-shield-x'),
    'text' => $session_percentage . '% ' . __t('guvenli'),
    'suggestion' => $session_percentage >= 80 ? __t('session_guvenlik_ayarlari_mukemmel') : __t('session_guvenlik_ayarlari_iyilestirilebilir'),
    'score' => $session_score
];
$total_checks++;
$security_score += $session_score; // Session skorunu genel skora ekle

// 7. PHP sürüm kontrolü
$php_version = PHP_VERSION;
$min_php_version = '8.0.0'; // Önerilen minimum sürüm
$php_status = [
    'type' => 'success',
    'icon' => 'bi-check-circle-fill',
    'text' => __t('guncel') . ' (' . $php_version . ')',
    'suggestion' => __t('php_surumu_guncel_ve_guvenli'),
    'score' => 10
];
$total_checks++;
if (version_compare($php_version, $min_php_version, '>=')) {
    $security_score += 10;
} else if (version_compare($php_version, '7.4.0', '>=')) {
    $php_status = [
        'type' => 'warning',
        'icon' => 'bi-exclamation-triangle-fill',
        'text' => __t('orta') . ' (' . $php_version . ')',
        'suggestion' => __t('php_surumu_guncel_degil_8_0_onerilir'),
        'score' => 5
    ];
    $security_score += 5;
} else {
    $php_status = [
        'type' => 'danger',
        'icon' => 'bi-x-circle-fill',
        'text' => __t('eski') . ' (' . $php_version . ')',
        'suggestion' => __t('php_surumu_cok_eski_guvenlik_riski_olusturabilir'),
        'score' => 0
    ];
}

// Güvenlik skoru yüzdesi
$security_percentage = $total_checks > 0 ? round(($security_score / ($total_checks * 20)) * 100) : 0; // Her kontrolün max skoru 20 varsayıldı

?>
<!DOCTYPE html>
<html lang="<?= htmlspecialchars($_SESSION['language'] ?? 'tr') ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo Yönetim - <?= temizle($sayfa_baslik) ?></title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            background-color: #f4f7f6; /* Hafif gri arka plan */
        }
        .security-check-card {
            border-radius: 0.75rem;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.05);
            transition: transform 0.2s ease-in-out;
        }
        .security-check-card:hover {
            transform: translateY(-5px);
        }
        .security-check-card .card-header {
            border-top-left-radius: 0.75rem;
            border-top-right-radius: 0.75rem;
            font-weight: bold;
        }
        .security-check-card .card-body {
            display: flex;
            flex-direction: column;
            justify-content: space-between;
            min-height: 120px; /* Kartların eşit yüksekliğini sağlamak için */
        }
        .status-icon-large {
            font-size: 2.5rem;
            margin-right: 15px;
            vertical-align: middle;
        }
        .status-badge {
            font-size: 0.9em;
            padding: 0.4em 0.8em;
            border-radius: 0.5rem;
        }
        .suggestion-text {
            font-size: 0.85em;
            color: #6c757d;
            margin-top: 8px;
        }
    </style>
</head>
<body>
    <div class="page">
        <?php include __DIR__ . '/../includes/navbar.php'; ?>
        <div class="container mt-4">
            <div class="p-4 mb-4 bg-white rounded-3 shadow-sm">
                <h1 class="display-5 fw-bold text-center text-primary"><i class="bi bi-shield-lock-fill me-3"></i><?= temizle($sayfa_baslik) ?></h1>
                <p class="fs-5 text-muted text-center">Sisteminizin temel güvenlik yapılandırmalarını ve potansiyel sorunları gözden geçirin.</p>
            </div>

            <!-- Mesajlar -->
            <?php if (!empty($_SESSION['mesaj'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="bi bi-check-circle-fill me-2"></i> <?= $_SESSION['mesaj'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['mesaj']); ?>
            <?php endif; ?>
            <?php if (!empty($_SESSION['hata'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i> <?= $_SESSION['hata'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['hata']); ?>
            <?php endif; ?>

            <?php foreach ($messages as $msg): // install.php silme sonucu gibi özel mesajlar ?>
                <div class="alert alert-<?= $msg['type'] ?> alert-dismissible fade show" role="alert">
                    <i class="bi bi-info-circle-fill me-2"></i>
                    <?= $msg['text'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endforeach; ?>

            <!-- Güvenlik Skoru -->
            <div class="card security-check-card border-<?= $security_percentage >= 80 ? 'success' : ($security_percentage >= 60 ? 'warning' : 'danger') ?> mb-4">
                <div class="card-header bg-<?= $security_percentage >= 80 ? 'success' : ($security_percentage >= 60 ? 'warning' : 'danger') ?> text-white">
                    <i class="bi bi-shield-lock-fill me-2"></i> Genel Güvenlik Skoru
                </div>
                <div class="card-body d-flex align-items-center justify-content-between">
                    <div>
                        <h5 class="card-title">Sistem Güvenliği Puanı: <span class="badge bg-primary fs-5"><?= $security_percentage ?>%</span></h5>
                        <p class="card-text suggestion-text">Bu puan, sisteminizin genel güvenlik durumunu yansıtır.</p>
                    </div>
                    <div class="progress" style="width: 150px; height: 20px;">
                        <div class="progress-bar bg-<?= $security_percentage >= 80 ? 'success' : ($security_percentage >= 60 ? 'warning' : 'danger') ?>" role="progressbar" style="width: <?= $security_percentage ?>%;" aria-valuenow="<?= $security_percentage ?>" aria-valuemin="0" aria-valuemax="100">
                            <?= $security_percentage ?>%
                        </div>
                    </div>
                </div>
            </div>

            <!-- Kritik Güvenlik Uyarısı (install.php) -->
            <div class="card security-check-card border-<?= $install_status['type'] ?> mb-4">
                <div class="card-header bg-<?= $install_status['type'] ?> text-white">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i> Install.php Dosya Durumu
                </div>
                <div class="card-body d-flex align-items-center">
                    <i class="bi <?= $install_status['icon'] ?> status-icon-large text-<?= $install_status['type'] ?>"></i>
                    <div>
                        <h5 class="card-title"><?= $install_status['text'] ?></h5>
                        <p class="card-text suggestion-text"><?= $install_status['suggestion'] ?></p>
                        <?php if ($install_file_exists): ?>
                            <form method="POST" class="mt-3">
                                <input type="hidden" name="csrf_token" value="<?= csrf_token_olustur() ?>">
                                <button type="submit" name="delete_install" class="btn btn-danger btn-sm"
                                        onclick="return confirm('<?= __t('install_php_dosyasini_silmek_istediginizden_emin_misiniz') ?>')">
                                    <i class="bi bi-trash me-1"></i> <?= __t('install_php_dosyasini_sil') ?>
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <h3 class="mb-4 text-center text-secondary">Diğer Önemli Kontroller</h3>
            <div class="row g-4">
                <?php
                // Her bir kontrol için dinamik kart oluşturma
                $checks_to_display = [
                    'https_usage' => ['label' => __t('https_kullanimi'), 'status_data' => $https_status, 'score' => 25],
                    'php_version' => ['label' => 'PHP Sürümü', 'status_data' => $php_status, 'score' => 10],
                    'config_perms' => ['label' => __t('config_php_dosya_izinleri'), 'status_data' => $config_status, 'score' => 20],
                    'error_log' => ['label' => __t('hata_gunlugu_durumu'), 'status_data' => $error_log_status, 'score' => 15],
                    'backups_dir' => ['label' => __t('yedekler_dizini_yazilabilirligi'), 'status_data' => $backups_status, 'score' => 10],
                    'session_security' => ['label' => 'Session Güvenliği', 'status_data' => $session_status, 'score' => 100]
                ];

                foreach ($checks_to_display as $key => $check):
                ?>
                    <div class="col-md-6 col-lg-4">
                        <div class="card security-check-card border-<?= $check['status_data']['type'] ?>">
                            <div class="card-header bg-<?= $check['status_data']['type'] ?> text-white">
                                <i class="bi <?= $check['status_data']['icon'] ?> me-2"></i> <?= $check['label'] ?>
                            </div>
                            <div class="card-body">
                                <h5 class="card-title"><?= $check['status_data']['text'] ?> <span class="badge bg-primary status-badge"><?= $check['status_data']['score'] ?>/<?= $check['score'] ?></span></h5>
                                <p class="card-text suggestion-text"><?= $check['status_data']['suggestion'] ?></p>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Session Güvenlik Detay Raporu -->
            <div class="card security-check-card mb-4 mt-4">
                <div class="card-header bg-info text-white">
                    <i class="bi bi-info-circle-fill me-2"></i> Oturum Güvenlik Detay Raporu
                </div>
                <div class="card-body">
                    <?= session_guvenlik_html_rapor() ?>
                </div>
            </div>

            <!-- Session Güvenlik İyileştirme -->
            <?php if ($session_percentage < 100): ?>
                <div class="card security-check-card border-warning mb-4">
                    <div class="card-header bg-warning text-dark">
                        <i class="bi bi-tools me-2"></i> Session Güvenlik İyileştirme
                    </div>
                    <div class="card-body">
                        <p class="card-text">Session güvenlik ayarlarını otomatik olarak iyileştirmek için aşağıdaki butona tıklayın.</p>
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle-fill me-2"></i> Bu işlem mevcut oturumunuzu sonlandıracak ve yeniden giriş yapmanız gerekecektir.
                        </div>
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?= csrf_token_olustur() ?>">
                            <button type="submit" name="improve_session" class="btn btn-primary mt-2"
                                    onclick="return confirm('Session güvenlik ayarlarını iyileştirmek istediğinizden emin misiniz? Bu işlem oturumunuzu sonlandıracaktır.')">
                                <i class="bi bi-shield-check me-2"></i> Session Güvenliğini İyileştir
                            </button>
                        </form>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Genel Güvenlik Önerileri -->
            <div class="card security-check-card border-primary mb-4">
                <div class="card-header bg-primary text-white">
                    <i class="bi bi-lightbulb-fill me-2"></i> Genel Güvenlik Önerileri
                </div>
                <div class="card-body">
                    <h4>Yüksek Öncelikli Öneriler</h4>
                    <ul class="list-unstyled">
                        <?php if ($install_file_exists): ?>
                            <li><i class="bi bi-x-circle-fill text-danger me-2"></i> <strong><?= __t('install_php_dosyasini_silin') ?></strong> - <?= __t('guvenlik_riski_olusturuyor') ?></li>
                        <?php endif; ?>
                        <?php if (!$is_https): ?>
                            <li><i class="bi bi-exclamation-triangle-fill text-danger me-2"></i> <strong><?= __t('https_kullanin') ?></strong> - <?= __t('ssl_sertifikasi_kurun') ?></li>
                        <?php endif; ?>
                        <?php if ($config_chmod !== '0644' && $config_chmod !== '0600'): ?>
                            <li><i class="bi bi-exclamation-triangle-fill text-warning me-2"></i> <strong><?= __t('config_php_izinlerini_duzeltin') ?></strong> - <?= __t('0644_veya_0600_olarak_ayarlayin') ?></li>
                        <?php endif; ?>
                        <?php if ($session_percentage < 100): ?>
                            <li><i class="bi bi-exclamation-triangle-fill text-warning me-2"></i> <strong><?= __t('session_guvenligini_iyilestirin') ?></strong> - <?= __t('yukaridaki_butonu_kullanin') ?></li>
                        <?php endif; ?>
                    </ul>

                    <h4 class="mt-4">Orta Öncelikli Öneriler</h4>
                    <ul class="list-unstyled">
                        <li><i class="bi bi-info-circle-fill text-info me-2"></i> <strong><?= __t('duzenli_yedekleme') ?></strong> - <?= __t('otomatik_yedekleme_sistemi_kurun') ?></li>
                        <li><i class="bi bi-info-circle-fill text-info me-2"></i> <strong><?= __t('hata_loglarini_kontrol_edin') ?></strong> - <?= __t('duzenli_olarak_inceleyin') ?></li>
                        <li><i class="bi bi-info-circle-fill text-info me-2"></i> <strong><?= __t('guclu_sifreler') ?></strong> - <?= __t('tum_kullanicilar_icin_guclu_sifre_zorunlulugu') ?></li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
    
    <?php include __DIR__ . '/../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
