<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi();

$sayfa_baslik = "Hatırlatıcı Sistemi";

// Hatırlatıcı ekleme işlemi
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['ekle'])) {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = "Geçersiz form gönderimi!";
        header("Location: hatirlatici.php");
        exit();
    }
    
    // Form verilerini al
    $baslik = temizle($_POST['baslik'] ?? '');
    $aciklama = temizle($_POST['aciklama'] ?? '');
    $hatirlatma_tarihi = filter_input(INPUT_POST, 'hatirlatma_tarihi', FILTER_SANITIZE_SPECIAL_CHARS);
    $oncelik = filter_input(INPUT_POST, 'oncelik', FILTER_SANITIZE_SPECIAL_CHARS);
    $cari_id = filter_input(INPUT_POST, 'cari_id', FILTER_VALIDATE_INT) ?: null;
    $borc_id = filter_input(INPUT_POST, 'borc_id', FILTER_VALIDATE_INT) ?: null;
    
    // Veri doğrulama
    $hatalar = [];
    if (empty($baslik)) $hatalar[] = "Başlık alanı zorunludur.";
    if (empty($hatirlatma_tarihi) || !preg_match("/^\d{4}-\d{2}-\d{2}$/", $hatirlatma_tarihi)) $hatalar[] = "Geçerli bir hatırlatma tarihi giriniz (YYYY-AA-GG).";
    if (!in_array($oncelik, ['dusuk', 'normal', 'yuksek'])) $hatalar[] = "Geçerli bir öncelik seçiniz.";
    
    if (empty($hatalar)) {
        try {
            $stmt = $db->prepare("
                INSERT INTO hatirlaticilar (sirket_id, baslik, aciklama, hatirlatma_tarihi, oncelik, cari_id, borc_id, kullanici_id, olusturma_tarihi, durum)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), 'aktif')
            ");
            
            $stmt->execute([
                $aktif_sirket_id,
                $baslik,
                $aciklama,
                $hatirlatma_tarihi,
                $oncelik,
                $cari_id,
                $borc_id,
                $_SESSION['kullanici_id']
            ]);
            
            $_SESSION['mesaj'] = "Hatırlatıcı başarıyla eklendi.";
            header("Location: hatirlatici.php");
            exit();
            
        } catch (PDOException $e) {
            error_log("Hatırlatıcı ekleme hatası: " . $e->getMessage());
            $_SESSION['hata'] = "Hatırlatıcı eklenirken bir hata oluştu.";
        }
    } else {
        $_SESSION['form_hatalari'] = $hatalar;
        $_SESSION['form_verileri'] = $_POST;
    }
    
    header("Location: hatirlatici.php");
    exit();
}

// Hatırlatıcı durumu güncelleme
if (isset($_GET['islem']) && isset($_GET['id'])) {
    $islem = $_GET['islem'];
    $id = filter_var($_GET['id'], FILTER_VALIDATE_INT);
    
    if ($id && in_array($islem, ['tamamla', 'sil'])) {
        try {
            if ($islem == 'tamamla') {
                $stmt = $db->prepare("UPDATE hatirlaticilar SET durum = 'tamamlandi', guncelleme_tarihi = NOW() WHERE id = ? AND kullanici_id = ?");
                $stmt->execute([$id, $_SESSION['kullanici_id']]);
                $_SESSION['mesaj'] = "Hatırlatıcı tamamlandı olarak işaretlendi.";
            } else {
                $stmt = $db->prepare("DELETE FROM hatirlaticilar WHERE id = ? AND kullanici_id = ?");
                $stmt->execute([$id, $_SESSION['kullanici_id']]);
                $_SESSION['mesaj'] = "Hatırlatıcı silindi.";
            }
        } catch (PDOException $e) {
            error_log("Hatırlatıcı güncelleme hatası: " . $e->getMessage());
            $_SESSION['hata'] = "İşlem sırasında bir hata oluştu.";
        }
        
        header("Location: hatirlatici.php");
        exit();
    }
}

$aktif_sirket_id = $_SESSION['aktif_sirket_id'] ?? 0;
// Hatırlatıcıları getir
try {
    // Aktif hatırlatıcılar
    $stmt_aktif = $db->prepare("
        SELECT h.*, c.ad_soyad as cari_adi, 
               b.tur as borc_tur, b.tutar as borc_tutar, b.aciklama as borc_aciklama
        FROM hatirlaticilar h
        LEFT JOIN cariler c ON h.cari_id = c.id AND c.sirket_id = ?
        LEFT JOIN borclar b ON h.borc_id = b.id AND b.sirket_id = ?
        WHERE h.kullanici_id = ? AND h.durum = 'aktif'
        ORDER BY h.hatirlatma_tarihi ASC
    ");
    $stmt_aktif->execute([$aktif_sirket_id, $aktif_sirket_id, $_SESSION['kullanici_id']]);
    $aktif_hatirlaticilar = $stmt_aktif->fetchAll(PDO::FETCH_ASSOC);
    
    // Tamamlanan hatırlatıcılar
    $stmt_tamamlanan = $db->prepare("
        SELECT h.*, c.ad_soyad as cari_adi
        FROM hatirlaticilar h
        LEFT JOIN cariler c ON h.cari_id = c.id AND c.sirket_id = ?
        WHERE h.kullanici_id = ? AND h.durum = 'tamamlandi'
        ORDER BY h.guncelleme_tarihi DESC
        LIMIT 10
    ");
    $stmt_tamamlanan->execute([$aktif_sirket_id, $_SESSION['kullanici_id']]);
    $tamamlanan_hatirlaticilar = $stmt_tamamlanan->fetchAll(PDO::FETCH_ASSOC);
    
    // Sadece aktif şirkete ait cariler
    $cariler = $db->prepare("SELECT id, ad_soyad FROM cariler WHERE sirket_id = ? ORDER BY ad_soyad");
    $cariler->execute([$aktif_sirket_id]);
    $cariler = $cariler->fetchAll(PDO::FETCH_ASSOC);
    
    // Vadesi yaklaşan borçlar
    $bugun = date('Y-m-d');
    $bir_ay_sonra = date('Y-m-d', strtotime('+30 days'));
    
    $stmt_borclar = $db->prepare("
        SELECT b.id, b.tur, (b.tutar + b.toplam_kdv - b.indirim_tutari) as toplam_tutar, b.vade_tarihi, b.aciklama, c.ad_soyad as cari_adi,
               DATEDIFF(b.vade_tarihi, ?) as kalan_gun
        FROM borclar b
        JOIN cariler c ON b.cari_id = c.id AND c.sirket_id = b.sirket_id
        WHERE b.durum != 'odendi' AND b.vade_tarihi BETWEEN ? AND ? AND b.sirket_id = ?
        ORDER BY b.vade_tarihi ASC
        LIMIT 20
    ");
    $stmt_borclar->execute([$bugun, $bugun, $bir_ay_sonra, $aktif_sirket_id]);
    $vadesi_yaklasan_borclar = $stmt_borclar->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Hatırlatıcı veri çekme hatası: " . $e->getMessage());
    $_SESSION['hata'] = "Veriler yüklenirken bir hata oluştu.";
    $aktif_hatirlaticilar = [];
    $tamamlanan_hatirlaticilar = [];
    $cariler = [];
    $vadesi_yaklasan_borclar = [];
}

$csrf_token = csrf_token_olustur();
$form_hatalari = $_SESSION['form_hatalari'] ?? [];
$form_verileri = $_SESSION['form_verileri'] ?? [];
unset($_SESSION['form_hatalari'], $_SESSION['form_verileri']);

// Hatırlatıcı tablosunu oluştur (eğer yoksa)
try {
    $db->exec("
        CREATE TABLE IF NOT EXISTS hatirlaticilar (
            id INT AUTO_INCREMENT PRIMARY KEY,
            baslik VARCHAR(255) NOT NULL,
            aciklama TEXT,
            hatirlatma_tarihi DATE NOT NULL,
            oncelik ENUM('dusuk', 'normal', 'yuksek') NOT NULL DEFAULT 'normal',
            cari_id INT,
            borc_id INT,
            kullanici_id INT NOT NULL,
            olusturma_tarihi DATETIME NOT NULL,
            guncelleme_tarihi DATETIME,
            durum ENUM('aktif', 'tamamlandi') NOT NULL DEFAULT 'aktif',
            FOREIGN KEY (cari_id) REFERENCES cariler(id) ON DELETE SET NULL,
            FOREIGN KEY (borc_id) REFERENCES borclar(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ");
} catch (PDOException $e) {
    error_log("Hatırlatıcı tablosu oluşturma hatası: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo Yönetim - <?= temizle($sayfa_baslik) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        .hatirlatici-card {
            transition: transform 0.2s;
            border-left: 4px solid #6c757d;
        }
        .hatirlatici-card:hover {
            transform: translateY(-3px);
        }
        .hatirlatici-dusuk { border-left-color: #0dcaf0; }
        .hatirlatici-normal { border-left-color: #6c757d; }
        .hatirlatici-yuksek { border-left-color: #dc3545; }
        .badge-dusuk { background-color: #0dcaf0; }
        .badge-normal { background-color: #6c757d; }
        .badge-yuksek { background-color: #dc3545; }
        .vade-gecmis { background-color: rgba(220, 53, 69, 0.1); }
        .vade-yakin { background-color: rgba(255, 193, 7, 0.1); }
        
        /* Mobil uyumluluk */
        @media (max-width: 768px) {
            .card-title { font-size: 1.1rem; }
            .hatirlatici-card { margin-bottom: 15px; }
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h2><i class="bi bi-bell"></i> <?= temizle($sayfa_baslik) ?></h2>
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#yeniHatirlaticiModal">
                <i class="bi bi-plus-circle"></i> Yeni Hatırlatıcı
            </button>
        </div>
        <hr>
        
        <?php if (isset($_SESSION['mesaj'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle-fill me-2"></i> <?= htmlspecialchars($_SESSION['mesaj']) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['mesaj']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['hata'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle-fill me-2"></i> <?= htmlspecialchars($_SESSION['hata']) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['hata']); ?>
        <?php endif; ?>
        
        <div class="row">
            <div class="col-lg-8">
                <!-- Aktif Hatırlatıcılar -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-light d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0"><i class="bi bi-bell-fill me-2 text-primary"></i>Aktif Hatırlatıcılar</h5>
                        <span class="badge bg-primary"><?= count($aktif_hatirlaticilar) ?></span>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($aktif_hatirlaticilar)): ?>
                            <div class="text-center p-4">
                                <i class="bi bi-bell-slash display-1 text-muted"></i>
                                <p class="text-muted mt-2">Aktif hatırlatıcı bulunmamaktadır.</p>
                                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#yeniHatirlaticiModal">
                                    <i class="bi bi-plus-circle"></i> İlk Hatırlatıcıyı Ekle
                                </button>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th style="width: 40px;">Öncelik</th>
                                            <th>Başlık & Açıklama</th>
                                            <th style="width: 120px;">Tarih</th>
                                            <th style="width: 150px;">İlişkili</th>
                                            <th style="width: 120px;" class="text-center">İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($aktif_hatirlaticilar as $hatirlatici): 
                                            $kalan_gun = (strtotime($hatirlatici['hatirlatma_tarihi']) - time()) / (60 * 60 * 24);
                                            $gecmis_mi = $kalan_gun < 0;
                                            $bugun_mu = $kalan_gun >= 0 && $kalan_gun < 1;
                                            $yakin_mi = $kalan_gun >= 1 && $kalan_gun <= 3;
                                            
                                            $row_class = '';
                                            if ($gecmis_mi) $row_class = 'table-danger';
                                            elseif ($bugun_mu) $row_class = 'table-warning';
                                            elseif ($yakin_mi) $row_class = 'table-info';
                                        ?>
                                        <tr class="<?= $row_class ?>">
                                            <td class="align-middle text-center">
                                                <?php 
                                                $oncelik_icon = '';
                                                $oncelik_class = '';
                                                switch($hatirlatici['oncelik']) {
                                                    case 'dusuk': 
                                                        $oncelik_icon = 'bi-arrow-down-circle';
                                                        $oncelik_class = 'text-info';
                                                        break;
                                                    case 'normal': 
                                                        $oncelik_icon = 'bi-dash-circle';
                                                        $oncelik_class = 'text-secondary';
                                                        break;
                                                    case 'yuksek': 
                                                        $oncelik_icon = 'bi-arrow-up-circle';
                                                        $oncelik_class = 'text-danger';
                                                        break;
                                                }
                                                ?>
                                                <i class="bi <?= $oncelik_icon ?> <?= $oncelik_class ?>" title="<?= ucfirst($hatirlatici['oncelik']) ?> Öncelik"></i>
                                            </td>
                                            <td class="align-middle">
                                                <div class="fw-bold text-dark"><?= temizle($hatirlatici['baslik']) ?></div>
                                                <?php if (!empty($hatirlatici['aciklama'])): ?>
                                                    <small class="text-muted"><?= temizle($hatirlatici['aciklama']) ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td class="align-middle">
                                                <div class="d-flex flex-column">
                                                    <span class="fw-bold"><?= date('d.m.Y', strtotime($hatirlatici['hatirlatma_tarihi'])) ?></span>
                                                    <?php if ($gecmis_mi): ?>
                                                        <small class="badge bg-danger">Gecikmiş</small>
                                                    <?php elseif ($bugun_mu): ?>
                                                        <small class="badge bg-warning text-dark">Bugün</small>
                                                    <?php elseif ($yakin_mi): ?>
                                                        <small class="badge bg-info">Yakında</small>
                                                    <?php else: ?>
                                                        <small class="text-muted"><?= ceil($kalan_gun) ?> gün kaldı</small>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            <td class="align-middle">
                                                <?php if (!empty($hatirlatici['cari_adi'])): ?>
                                                    <div class="mb-1">
                                                        <i class="bi bi-person text-primary me-1"></i>
                                                        <small><?= temizle($hatirlatici['cari_adi']) ?></small>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if (!empty($hatirlatici['borc_aciklama'])): ?>
                                                    <div>
                                                        <i class="bi bi-cash text-success me-1"></i>
                                                        <small><?= number_format($hatirlatici['borc_tutar'], 0, ',', '.') . ' ' . getParaBirimiSembolu() ?></small>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if (empty($hatirlatici['cari_adi']) && empty($hatirlatici['borc_aciklama'])): ?>
                                                    <small class="text-muted">-</small>
                                                <?php endif; ?>
                                            </td>
                                            <td class="align-middle text-center">
                                                <div class="btn-group btn-group-sm" role="group">
                                                    <a href="?islem=tamamla&id=<?= $hatirlatici['id'] ?>" 
                                                       class="btn btn-outline-success" 
                                                       title="Tamamla"
                                                       onclick="return confirm('Bu hatırlatıcıyı tamamlandı olarak işaretlemek istediğinize emin misiniz?')">
                                                        <i class="bi bi-check-circle"></i>
                                                    </a>
                                                    <a href="?islem=sil&id=<?= $hatirlatici['id'] ?>" 
                                                       class="btn btn-outline-danger" 
                                                       title="Sil"
                                                       onclick="return confirm('Bu hatırlatıcıyı silmek istediğinize emin misiniz?')">
                                                        <i class="bi bi-trash"></i>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Tamamlanan Hatırlatıcılar -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0"><i class="bi bi-check-circle-fill me-2 text-success"></i>Tamamlanan Hatırlatıcılar</h5>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($tamamlanan_hatirlaticilar)): ?>
                            <div class="alert alert-info m-3">
                                <i class="bi bi-info-circle-fill me-2"></i> Tamamlanan hatırlatıcı bulunmamaktadır.
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Başlık</th>
                                            <th>Hatırlatma Tarihi</th>
                                            <th>Cari</th>
                                            <th>Tamamlanma Tarihi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($tamamlanan_hatirlaticilar as $hatirlatici): ?>
                                        <tr>
                                            <td><?= temizle($hatirlatici['baslik']) ?></td>
                                            <td><?= date('d.m.Y', strtotime($hatirlatici['hatirlatma_tarihi'])) ?></td>
                                            <td><?= temizle($hatirlatici['cari_adi'] ?? '-') ?></td>
                                            <td><?= date('d.m.Y H:i', strtotime($hatirlatici['guncelleme_tarihi'])) ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4">
                <!-- Vadesi Yaklaşan Borçlar/Alacaklar -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0"><i class="bi bi-calendar-check me-2 text-warning"></i>Vadesi Yaklaşan İşlemler</h5>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($vadesi_yaklasan_borclar)): ?>
                            <div class="alert alert-info m-3">
                                <i class="bi bi-info-circle-fill me-2"></i> Vadesi yaklaşan borç/alacak bulunmamaktadır.
                            </div>
                        <?php else: ?>
                            <div class="list-group list-group-flush">
                                <?php foreach ($vadesi_yaklasan_borclar as $borc):
                                    $kalan_gun = $borc['kalan_gun'];
                                    $vade_class = '';
                                    if ($kalan_gun !== null) {
                                        if ($kalan_gun < 0) $vade_class = 'vade-gecmis';
                                        elseif ($kalan_gun <= 7) $vade_class = 'vade-yakin';
                                    }
                                ?>
                                <div class="list-group-item <?= $vade_class ?>">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <h6 class="mb-1">
                                                <?= temizle($borc['cari_adi']) ?>
                                                <span class="badge <?= $borc['tur'] == 'satis' ? 'bg-success' : 'bg-danger' ?>">
                                                    <?= $borc['tur'] == 'satis' ? 'Alacak' : 'Borç' ?>
                                                </span>
                                            </h6>
                                            <p class="mb-1 small text-muted"><?= temizle($borc['aciklama']) ?></p>
                                            <small>
                                                <i class="bi bi-calendar me-1"></i> <?= date('d.m.Y', strtotime($borc['vade_tarihi'])) ?>
                                                <?php if ($kalan_gun < 0): ?>
                                                    <span class="badge bg-danger ms-1"><?= abs($kalan_gun) ?> gün geçti</span>
                                                <?php elseif ($kalan_gun == 0): ?>
                                                    <span class="badge bg-warning text-dark ms-1">Bugün</span>
                                                <?php else: ?>
                                                    <span class="badge bg-info text-dark ms-1"><?= $kalan_gun ?> gün kaldı</span>
                                                <?php endif; ?>
                                            </small>
                                        </div>
                                        <div class="text-end">
                                            <div class="<?= $borc['tur'] == 'satis' ? 'text-success' : 'text-danger' ?> fw-bold">
                                                <?= number_format($borc['toplam_tutar'], 2, ',', '.') . ' ' . getParaBirimiSembolu() ?>
                                            </div>
                                            <button class="btn btn-sm btn-outline-primary mt-1" 
                                                    onclick="hatirlat(<?= $borc['id'] ?>, '<?= addslashes($borc['cari_adi']) ?>', '<?= $borc['tur'] ?>', '<?= $borc['vade_tarihi'] ?>')">
                                                <i class="bi bi-bell"></i> Hatırlat
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Hızlı Hatırlatıcı Oluştur -->
                <div class="card shadow-sm">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0"><i class="bi bi-lightning-fill me-2 text-warning"></i>Hızlı Hatırlatıcı</h5>
                    </div>
                    <div class="card-body">
                        <form method="post" class="needs-validation" novalidate>
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            <input type="hidden" name="ekle" value="1">
                            
                            <div class="mb-3">
                                <label for="baslik" class="form-label">Başlık <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="baslik" name="baslik" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="hatirlatma_tarihi" class="form-label">Hatırlatma Tarihi <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="hatirlatma_tarihi" name="hatirlatma_tarihi" required
                                       min="<?= date('Y-m-d') ?>" value="<?= date('Y-m-d', strtotime('+1 day')) ?>">
                            </div>
                            
                            <div class="mb-3">
                                <label for="oncelik" class="form-label">Öncelik</label>
                                <select class="form-select" id="oncelik" name="oncelik">
                                    <option value="dusuk">Düşük</option>
                                    <option value="normal" selected>Normal</option>
                                    <option value="yuksek">Yüksek</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="aciklama" class="form-label">Açıklama</label>
                                <textarea class="form-control" id="aciklama" name="aciklama" rows="2"></textarea>
                            </div>
                            
                            <div class="d-grid">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-plus-circle me-2"></i> Hatırlatıcı Ekle
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Yeni Hatırlatıcı Modal -->
    <div class="modal fade" id="yeniHatirlaticiModal" tabindex="-1" aria-labelledby="yeniHatirlaticiModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="yeniHatirlaticiModalLabel">Yeni Hatırlatıcı Ekle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form method="post" id="hatirlaticiForm" class="needs-validation" novalidate>
                        <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                        <input type="hidden" name="ekle" value="1">
                        
                        <div class="row mb-3">
                            <div class="col-md-8">
                                <label for="modal_baslik" class="form-label">Başlık <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="modal_baslik" name="baslik" required>
                            </div>
                            <div class="col-md-4">
                                <label for="modal_oncelik" class="form-label">Öncelik</label>
                                <select class="form-select" id="modal_oncelik" name="oncelik">
                                    <option value="dusuk">Düşük</option>
                                    <option value="normal" selected>Normal</option>
                                    <option value="yuksek">Yüksek</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="modal_hatirlatma_tarihi" class="form-label">Hatırlatma Tarihi <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="modal_hatirlatma_tarihi" name="hatirlatma_tarihi" required
                                       min="<?= date('Y-m-d') ?>" value="<?= date('Y-m-d', strtotime('+1 day')) ?>">
                            </div>
                            <div class="col-md-6">
                                <label for="modal_cari_id" class="form-label">İlgili Cari</label>
                                <select class="form-select" id="modal_cari_id" name="cari_id">
                                    <option value="">-- Cari Seçin --</option>
                                    <?php foreach ($cariler as $cari): ?>
                                        <option value="<?= $cari['id'] ?>"><?= temizle($cari['ad_soyad']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="modal_borc_id" class="form-label">İlgili Borç/Alacak</label>
                            <select class="form-select" id="modal_borc_id" name="borc_id" disabled>
                                <option value="">-- Önce Cari Seçin --</option>
                            </select>
                            <small class="text-muted">Cari seçildikten sonra ilgili borç/alacaklar listelenecektir.</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="modal_aciklama" class="form-label">Açıklama</label>
                            <textarea class="form-control" id="modal_aciklama" name="aciklama" rows="3"></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="button" class="btn btn-primary" onclick="document.getElementById('hatirlaticiForm').submit();">
                        <i class="bi bi-plus-circle me-2"></i> Hatırlatıcı Ekle
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <?php include __DIR__ . '/../includes/footer.php'; ?>
    <script>
        // Form doğrulama
        document.addEventListener('DOMContentLoaded', function() {
            const forms = document.querySelectorAll('.needs-validation');
            
            Array.from(forms).forEach(form => {
                form.addEventListener('submit', event => {
                    if (!form.checkValidity()) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    
                    form.classList.add('was-validated');
                }, false);
            });
            
            // Cari seçildiğinde borçları getir
            const cariSelect = document.getElementById('modal_cari_id');
            const borcSelect = document.getElementById('modal_borc_id');
            
            cariSelect.addEventListener('change', function() {
                const cariId = this.value;
                
                if (cariId) {
                    borcSelect.disabled = true;
                    borcSelect.innerHTML = '<option value="">Yükleniyor...</option>';
                    
                    fetch(`get_borc_by_cari.php?cari_id=${cariId}`)
                        .then(response => response.json())
                        .then(data => {
                            borcSelect.innerHTML = '<option value="">-- Borç/Alacak Seçin --</option>';
                            
                            if (data.length > 0) {
                                data.forEach(borc => {
                                    const option = document.createElement('option');
                                    option.value = borc.id;
                                    option.textContent = `${borc.tur == 'satis' ? 'Alacak' : 'Borç'}: ${borc.aciklama} (${parseFloat(borc.tutar).toLocaleString('tr-TR')} ${'<?= getParaBirimiSembolu(); ?>'})`;
                                    borcSelect.appendChild(option);
                                });
                                borcSelect.disabled = false;
                            } else {
                                borcSelect.innerHTML = '<option value="">-- Borç/Alacak Bulunamadı --</option>';
                                borcSelect.disabled = true;
                            }
                        })
                        .catch(error => {
                            console.error('Borç getirme hatası:', error);
                            borcSelect.innerHTML = '<option value="">-- Hata Oluştu --</option>';
                            borcSelect.disabled = true;
                        });
                } else {
                    borcSelect.innerHTML = '<option value="">-- Önce Cari Seçin --</option>';
                    borcSelect.disabled = true;
                }
            });
        });
        
        // Vadesi yaklaşan borçlar için hatırlatıcı oluştur
        function hatirlat(borcId, cariAdi, borcTur, vadeTarihi) {
            const modal = new bootstrap.Modal(document.getElementById('yeniHatirlaticiModal'));
            
            // Form alanlarını doldur
            document.getElementById('modal_baslik').value = `${borcTur == 'satis' ? 'Tahsilat' : 'Ödeme'} Hatırlatması: ${cariAdi}`;
            document.getElementById('modal_hatirlatma_tarihi').value = vadeTarihi;
            document.getElementById('modal_oncelik').value = 'yuksek';
            document.getElementById('modal_aciklama').value = `${borcTur == 'satis' ? 'Tahsilat' : 'Ödeme'} vadesi: ${new Date(vadeTarihi).toLocaleDateString('tr-TR')}`;
            
            // Modalı göster
            modal.show();
        }
    </script>
</body>
</html>
