// Barkod okuyucu JavaScript modülü
// QuaggaJS kütüphanesi kullanılarak geliştirilmiştir

class BarkodOkuyucu {
    constructor() {
        this.isStarted = false;
        this.modal = null;
        this.onDetectedCallback = null;
        this.targetElement = null;
        this.kameraAlaniId = 'barkodKameraAlani'; // Dinamik olarak oluşturulan modal içindeki kamera alanı ID'si
    }

    // Barkod okuyucuyu başlat
    init(onDetectedCallback) {
        this.onDetectedCallback = onDetectedCallback;
        
        // Modal oluştur
        this.createModal();
        return true;
    }

    // Modal oluştur
    createModal() {
        const modalHtml = `
            <div class="modal fade" id="barkodOkuyucuModal" tabindex="-1" aria-labelledby="barkodOkuyucuModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="barkodOkuyucuModalLabel">
                                <i class="bi bi-upc-scan me-2"></i>Barkod Okuyucu
                            </h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            <div id="${this.kameraAlaniId}" style="width: 100%; height: 300px; background: #000; position: relative; border-radius: 0.375rem; overflow: hidden;">
                                <div class="text-center text-white p-4" id="kameraYukleniyor">
                                    <i class="bi bi-camera-video display-1"></i>
                                    <p class="mt-2">Kamera başlatılıyor...</p>
                                    <div class="spinner-border text-light" role="status">
                                        <span class="visually-hidden">Yükleniyor...</span>
                                    </div>
                                </div>
                                <div class="position-absolute top-50 start-50 translate-middle" id="barkodHedef" style="display: none;">
                                    <div style="width: 200px; height: 100px; border: 2px solid #00ff00; border-radius: 10px; background: rgba(0,255,0,0.1);"></div>
                                    <p class="text-center text-white mt-2">Barkodu bu alana getirin</p>
                                </div>
                            </div>
                            <div class="mt-3">
                                <div class="alert alert-info">
                                    <i class="bi bi-info-circle me-2"></i>
                                    <strong>Kullanım:</strong> Barkodu kamera görüş alanına getirin. Otomatik olarak tanınacaktır.
                                </div>
                                <div id="barkodSonuc" class="alert alert-success d-none">
                                    <i class="bi bi-check-circle me-2"></i>
                                    <strong>Barkod okundu:</strong> <span id="barkodDegeri"></span>
                                </div>
                                <div id="barkodHata" class="alert alert-danger d-none">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <span id="barkodHataMesaji"></span>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Kapat</button>
                            <button type="button" class="btn btn-primary" id="manuelBarkodBtn">
                                <i class="bi bi-keyboard me-2"></i>Manuel Giriş
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Modal'ı sayfaya ekle
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Modal referansını al
        this.modal = new bootstrap.Modal(document.getElementById('barkodOkuyucuModal'));
        
        // Event listener'ları ekle
        this.setupEventListeners();
    }

    // Event listener'ları ayarla
    setupEventListeners() {
        const modalElement = document.getElementById('barkodOkuyucuModal');
        
        // Modal açıldığında kamerayı başlat
        modalElement.addEventListener('shown.bs.modal', () => {
            this.startCamera();
        });
        
        // Modal kapandığında kamerayı durdur
        modalElement.addEventListener('hidden.bs.modal', () => {
            this.stopCamera();
        });

        // Manuel barkod girişi
        document.getElementById('manuelBarkodBtn').addEventListener('click', () => {
            this.showManualInput();
        });
    }

    // Kamerayı başlat
    startCamera() {
        if (this.isStarted) return;

        // Kamera izni kontrolü
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            this.showError('Bu tarayıcı kamera erişimini desteklemiyor.');
            return;
        }

        Quagga.init({
            inputStream: {
                name: "Live",
                type: "LiveStream",
                target: document.querySelector(`#${this.kameraAlaniId}`),
                constraints: {
                    width: { min: 640, ideal: 1280, max: 1920 },
                    height: { min: 480, ideal: 720, max: 1080 },
                    facingMode: "environment", // Arka kamera
                    aspectRatio: { min: 1, max: 2 }
                }
            },
            locator: {
                patchSize: "medium",
                halfSample: true
            },
            numOfWorkers: 2,
            frequency: 10,
            decoder: {
                readers: [
                    "code_128_reader",
                    "ean_reader",
                    "ean_8_reader",
                    "code_39_reader",
                    "code_39_vin_reader",
                    "codabar_reader",
                    "upc_reader",
                    "upc_e_reader",
                    "i2of5_reader"
                ]
            },
            locate: true
        }, (err) => {
            if (err) {
                console.error('Quagga başlatma hatası:', err);
                this.showError('Kamera başlatılamadı: ' + err.message);
                return;
            }
            
            console.log('Quagga başarıyla başlatıldı');
            Quagga.start();
            this.isStarted = true;
            
            // Yükleniyor mesajını gizle, hedef alanını göster
            document.getElementById('kameraYukleniyor').style.display = 'none';
            document.getElementById('barkodHedef').style.display = 'block';
        });

        // Barkod algılandığında
        Quagga.onDetected((data) => {
            const barkod = data.codeResult.code;
            console.log('Barkod algılandı:', barkod);
            
            // Sonucu göster
            this.showResult(barkod);
            
            // Callback'i çağır
            if (this.onDetectedCallback) {
                this.onDetectedCallback(barkod);
            }
            
            // Modal'ı kapat
            setTimeout(() => {
                this.modal.hide();
            }, 1500);
        });

        // Hata durumunda
        Quagga.onProcessed((result) => {
            const drawingCtx = Quagga.canvas.ctx.overlay;
            const drawingCanvas = Quagga.canvas.dom.overlay;

            if (result) {
                if (result.boxes) {
                    drawingCtx.clearRect(0, 0, parseInt(drawingCanvas.getAttribute("width")), parseInt(drawingCanvas.getAttribute("height")));
                    result.boxes.filter((box) => {
                        return box !== result.box;
                    }).forEach((box) => {
                        Quagga.ImageDebug.drawPath(box, {x: 0, y: 1}, drawingCtx, {color: "green", lineWidth: 2});
                    });
                }

                if (result.box) {
                    Quagga.ImageDebug.drawPath(result.box, {x: 0, y: 1}, drawingCtx, {color: "#00F", lineWidth: 2});
                }

                if (result.codeResult && result.codeResult.code) {
                    Quagga.ImageDebug.drawPath(result.line, {x: 'x', y: 'y'}, drawingCtx, {color: 'red', lineWidth: 3});
                }
            }
        });
    }

    // Kamerayı durdur
    stopCamera() {
        if (this.isStarted) {
            Quagga.stop();
            this.isStarted = false;
            
            // UI'yi sıfırla
            document.getElementById('kameraYukleniyor').style.display = 'block';
            document.getElementById('barkodHedef').style.display = 'none';
            this.hideResult();
            this.hideError();
        }
    }

    // Sonucu göster
    showResult(barkod) {
        document.getElementById('barkodDegeri').textContent = barkod;
        document.getElementById('barkodSonuc').classList.remove('d-none');
        document.getElementById('barkodHata').classList.add('d-none');
    }

    // Sonucu gizle
    hideResult() {
        document.getElementById('barkodSonuc').classList.add('d-none');
    }

    // Hata göster
    showError(mesaj) {
        document.getElementById('barkodHataMesaji').textContent = mesaj;
        document.getElementById('barkodHata').classList.remove('d-none');
        document.getElementById('barkodSonuc').classList.add('d-none');
        
        // Yükleniyor mesajını gizle
        document.getElementById('kameraYukleniyor').style.display = 'none';
    }

    // Hata gizle
    hideError() {
        document.getElementById('barkodHata').classList.add('d-none');
    }

    // Manuel barkod girişi göster
    showManualInput() {
        const barkod = prompt('Barkod numarasını manuel olarak girin:');
        if (barkod && barkod.trim()) {
            this.showResult(barkod.trim());
            if (this.onDetectedCallback) {
                this.onDetectedCallback(barkod.trim());
            }
            setTimeout(() => {
                this.modal.hide();
            }, 1000);
        }
    }

    // Modal'ı göster
    show() {
        if (this.modal) {
            this.modal.show();
        }
    }

    // Modal'ı gizle
    hide() {
        if (this.modal) {
            this.modal.hide();
        }
    }

    // Temizlik
    destroy() {
        this.stopCamera();
        if (this.modal) {
            const modalElement = document.getElementById('barkodOkuyucuModal');
            if (modalElement) {
                modalElement.remove();
            }
        }
    }
}

// Global barkod okuyucu instance'ı
window.BarkodOkuyucu = BarkodOkuyucu;


