<?php
require_once __DIR__ . '/../includes/config.php';
yetkili_mi('yonetici');

$sayfa_baslik = __("edit_quote");
$paraBirimi = getParaBirimiSembolu();
$aktif_sirket_id = $_SESSION['aktif_sirket_id'] ?? 0;

// Teklif ID kontrolü
$teklif_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$teklif_id) {
    $_SESSION['hata'] = __("invalid_quote_id");
    header("Location: teklifler.php");
    exit();
}

// Teklif bilgilerini getir
try {
    $stmt_teklif = $db->prepare("SELECT t.*, c.ad_soyad as cari_adi FROM teklifler t 
                                 LEFT JOIN cariler c ON t.cari_id = c.id 
                                 WHERE t.id = ? AND t.sirket_id = ?");
    $stmt_teklif->execute([$teklif_id, $aktif_sirket_id]);
    $teklif = $stmt_teklif->fetch(PDO::FETCH_ASSOC);
    
    if (!$teklif) {
        $_SESSION['hata'] = __("quote_not_found");
        header("Location: teklifler.php");
        exit();
    }
} catch (PDOException $e) {
    $_SESSION['hata'] = "Teklif bilgileri alınırken hata oluştu!";
    header("Location: teklifler.php");
    exit();
}

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!csrf_token_dogrula($_POST['csrf_token'] ?? '')) {
        $_SESSION['hata'] = __("invalid_form_submission");
    } else {
        $cari_id = filter_input(INPUT_POST, 'cari_id', FILTER_VALIDATE_INT);
        $teklif_tarihi = filter_input(INPUT_POST, 'teklif_tarihi', FILTER_SANITIZE_SPECIAL_CHARS);
        $gecerlilik_tarihi = filter_input(INPUT_POST, 'gecerlilik_tarihi', FILTER_SANITIZE_SPECIAL_CHARS);
        $genel_aciklama = temizle($_POST['genel_aciklama'] ?? '');
        
        // İndirim tutarını parse et
        $indirim_str = $_POST['indirim_tutari'] ?? '0';
        $indirim_str = str_replace('.', '', $indirim_str);
        $indirim_str = str_replace(',', '.', $indirim_str);
        $indirim_tutari = filter_var($indirim_str, FILTER_VALIDATE_FLOAT) ?: 0;

        // Kalem bilgilerini al
        $aciklamalar = $_POST['aciklama'] ?? [];
        $tutarlar = $_POST['tutar'] ?? [];
        $birlesik_aciklama = '';
        $toplam_tutar = 0;

        if (is_array($aciklamalar) && is_array($tutarlar)) {
            foreach ($tutarlar as $index => $tutar) {
                if (empty($aciklamalar[$index]) || empty($tutar)) continue;
                
                // Türkçe sayı formatını parse et
                $tutar_str = str_replace('.', '', $tutar);
                $tutar_str = str_replace(',', '.', $tutar_str);
                $tutar_float = filter_var($tutar_str, FILTER_VALIDATE_FLOAT);
                
                if ($tutar_float > 0) {
                    $toplam_tutar += $tutar_float;
                    $birlesik_aciklama .= trim($aciklamalar[$index]) . " (" . number_format($tutar_float, 2, ',', '.') . " " . $paraBirimi . ")\n";
                }
            }
        }
        
        $birlesik_aciklama = trim($birlesik_aciklama);
        
        // Genel açıklama varsa ekle
        if (!empty($genel_aciklama)) {
            $birlesik_aciklama = $genel_aciklama . "\n\n" . $birlesik_aciklama;
        }
        
        $toplam_kdv = 0; // Stoksuz işlemlerde KDV hesaplaması yapılmaz
        
        $hatalar = [];
        if (empty($cari_id)) $hatalar[] = "Müşteri seçimi zorunludur.";
        if (empty($teklif_tarihi)) $hatalar[] = "Teklif tarihi zorunludur.";
        if ($toplam_tutar <= 0) $hatalar[] = "Teklif için en az bir kalem eklemelisiniz.";

        if (empty($hatalar)) {
            try {
                $db->beginTransaction();
                
                // Teklifi güncelle
                $stmt_guncelle = $db->prepare("UPDATE teklifler SET 
                    cari_id = ?, teklif_tarihi = ?, gecerlilik_tarihi = ?, 
                    tutar = ?, toplam_kdv = ?, indirim_tutari = ?, aciklama = ? 
                    WHERE id = ? AND sirket_id = ?");
                $stmt_guncelle->execute([
                    $cari_id, $teklif_tarihi, empty($gecerlilik_tarihi) ? null : $gecerlilik_tarihi,
                    $toplam_tutar, $toplam_kdv, $indirim_tutari, $birlesik_aciklama,
                    $teklif_id, $aktif_sirket_id
                ]);

                $db->commit();

                // Loglama
                $stmt_cari_ad = $db->prepare("SELECT ad_soyad FROM cariler WHERE id = ? AND sirket_id = ?");
                $stmt_cari_ad->execute([$cari_id, $aktif_sirket_id]);
                $cari_adi = $stmt_cari_ad->fetchColumn();
                $genel_toplam = $toplam_tutar - $indirim_tutari;
                $log_mesaji = sprintf(__('user_updated_quote_log'), $teklif_id, $cari_adi, number_format($genel_toplam, 2) . ' ' . $paraBirimi);
                sistem_log_ekle($db, 'guncelle', 'teklifler', $teklif_id, $teklif, ['yeni_veriler' => $_POST], $log_mesaji);

                $_SESSION['mesaj'] = sprintf(__('quote_updated_success'), $teklif_id);
                header("Location: teklifler.php");
                exit();
            } catch (Exception $e) {
                $db->rollBack();
                $_SESSION['hata'] = __('error_updating_quote') . ': ' . $e->getMessage();
            }
        } else {
            $_SESSION['form_hatalari'] = $hatalar;
            $_SESSION['form_verileri'] = $_POST;
        }
    }
    header("Location: teklifstoksuz_duzenle.php?id=" . $teklif_id);
    exit();
}

// Cariler sorgusu
$stmt_cariler = $db->prepare("SELECT id, ad_soyad FROM cariler WHERE sirket_id = ? ORDER BY ad_soyad");
$stmt_cariler->execute([$aktif_sirket_id]);
$cariler = $stmt_cariler->fetchAll(PDO::FETCH_ASSOC);

// Mevcut form verilerini al
$form_verileri = $_SESSION['form_verileri'] ?? [];
$form_hatalari = $_SESSION['form_hatalari'] ?? [];
unset($_SESSION['form_verileri'], $_SESSION['form_hatalari']);

// Teklif açıklamasını parse et
$teklif_aciklamasi = $teklif['aciklama'] ?? '';
$genel_aciklama = '';
$kalemler = [];

// Açıklamayı genel açıklama ve kalemlere ayır
if (!empty($teklif_aciklamasi)) {
    $satirlar = explode("\n", $teklif_aciklamasi);
    $kalem_satirlari = [];
    
    foreach ($satirlar as $satir) {
        $satir = trim($satir);
        if (empty($satir)) continue;
        
        // Kalem formatı: "Açıklama (Tutar PB)" şeklinde mi?
        if (preg_match('/^(.+?)\s*\(([0-9.,]+)\s*([A-Z]{2,3})\)$/', $satir, $matches)) {
            // Türkçe sayı formatını parse et
            $tutar_str = $matches[2];
            $tutar_str = str_replace('.', '', $tutar_str);
            $tutar_str = str_replace(',', '.', $tutar_str);
            
            $kalemler[] = [
                'aciklama' => trim($matches[1]),
                'tutar' => $tutar_str
            ];
        } else {
            // Genel açıklama olarak ekle
            if (empty($genel_aciklama)) {
                $genel_aciklama = $satir;
            } else {
                $genel_aciklama .= "\n" . $satir;
            }
        }
    }
}

// Eğer hiç kalem bulunamadıysa, tüm açıklamayı genel açıklama yap
if (empty($kalemler) && !empty($teklif_aciklamasi)) {
    $genel_aciklama = $teklif_aciklamasi;
    $kalemler = [['aciklama' => 'Genel hizmet', 'tutar' => $teklif['tutar']]];
}

$csrf_token = csrf_token_olustur();
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= temizle($sayfa_baslik) ?> - <?= __("app_name") ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <style>
        .hidden { display: none !important; }
        .toplam-alan { background: #fff3cd; border: 1px solid #ffeeba; border-radius: 0.375rem; }
        .is-invalid { border-color: #dc3545 !important; }
        .kalem-row { background: #f8f9fa; border-radius: 0.375rem; padding: 15px; margin-bottom: 15px; }
        .kalem-row:hover { background: #e9ecef; }
        .form-control:focus { border-color: #80bdff; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
    </style>
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="bi bi-pencil-square text-primary"></i> <?= temizle($sayfa_baslik) ?> #<?= $teklif_id ?></h2>
            <div>
                <a href="teklifler.php" class="btn btn-secondary"><i class="bi bi-list-ul"></i> <?= __("quote_list") ?></a>
                <a href="teklifstoksuz.php" class="btn btn-primary"><i class="bi bi-plus-circle"></i> <?= __("new_quote") ?></a>
            </div>
        </div>

        <!-- Hata ve Başarı Mesajları -->
        <?php if (!empty($form_hatalari)): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-exclamation-triangle-fill"></i>
                <strong>Hata!</strong>
                <ul class="mb-0 mt-2">
                    <?php foreach ($form_hatalari as $hata): ?>
                        <li><?= temizle($hata) ?></li>
                    <?php endforeach; ?>
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <form method="post" id="teklifForm">
            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
            
            <!-- Teklif Bilgileri -->
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-info-circle"></i> <?= __("quote_information") ?></h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="cari_id" class="form-label fw-bold"><?= __("customer") ?> <span class="text-danger">*</span></label>
                            <select name="cari_id" id="cari_id" class="form-select" required>
                                <option value=""><?= __("select_customer") ?></option>
                                <?php foreach ($cariler as $cari): ?>
                                    <option value="<?= $cari['id'] ?>" 
                                        <?= ($form_verileri['cari_id'] ?? $teklif['cari_id']) == $cari['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($cari['ad_soyad']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3 mb-3">
                            <label for="teklif_tarihi" class="form-label fw-bold"><?= __("quote_date") ?> <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" id="teklif_tarihi" name="teklif_tarihi" 
                                   value="<?= htmlspecialchars($form_verileri['teklif_tarihi'] ?? $teklif['teklif_tarihi'] ?? date('Y-m-d')) ?>" required>
                        </div>
                        <div class="col-md-3 mb-3">
                            <label for="gecerlilik_tarihi" class="form-label fw-bold"><?= __("valid_until") ?></label>
                            <input type="date" class="form-control" id="gecerlilik_tarihi" name="gecerlilik_tarihi" 
                                   value="<?= htmlspecialchars($form_verileri['gecerlilik_tarihi'] ?? $teklif['gecerlilik_tarihi'] ?? '') ?>">
                        </div>
                        <div class="col-12">
                            <label for="genel_aciklama" class="form-label fw-bold"><?= __("description") ?></label>
                            <textarea class="form-control" id="genel_aciklama" name="genel_aciklama" rows="3" 
                                      placeholder="<?= __("general_information_about_quote") ?>"><?= htmlspecialchars($genel_aciklama) ?></textarea>
                        </div>
                    </div>
                </div>
            </div>

            <!-- İşlem Kalemleri -->
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-success text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="bi bi-journal-text"></i> <?= __("transaction_items") ?></h5>
                    <button type="button" class="btn btn-light btn-sm" id="addKalemRow">
                        <i class="bi bi-plus-circle"></i> <?= __("add_item") ?>
                    </button>
                </div>
                <div class="card-body">
                    <div id="kalemRows">
                        <?php if (empty($kalemler)): ?>
                            <div class="kalem-row">
                                <div class="row">
                                    <div class="col-md-7">
                                        <label class="form-label fw-bold"><?= __("description") ?></label>
                                        <input type="text" class="form-control" name="aciklama[]" 
                                               placeholder="<?= __("e_g_consulting_services_project_management") ?>" required>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label fw-bold"><?= __("amount") ?> (<?= $paraBirimi ?>)</label>
                                        <input type="text" class="form-control tutar-input text-end" name="tutar[]" 
                                               placeholder="0,00" required>
                                    </div>
                                    <div class="col-md-1 d-flex align-items-end">
                                        <button type="button" class="btn btn-outline-danger remove-kalem-row" title="Kalemi Kaldır">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <?php foreach ($kalemler as $kalem): ?>
                                <div class="kalem-row">
                                    <div class="row">
                                        <div class="col-md-7">
                                            <label class="form-label fw-bold">Açıklama</label>
                                            <input type="text" class="form-control" name="aciklama[]" 
                                                   value="<?= htmlspecialchars($kalem['aciklama']) ?>" 
                                                   placeholder="Örn: Danışmanlık hizmeti, Proje yönetimi" required>
                                        </div>
                                        <div class="col-md-4">
                                            <label class="form-label fw-bold">Tutar (<?= $paraBirimi ?>)</label>
                                            <input type="text" class="form-control tutar-input text-end" name="tutar[]" 
                                                   value="<?= htmlspecialchars($kalem['tutar']) ?>" 
                                                   placeholder="0,00" required>
                                        </div>
                                        <div class="col-md-1 d-flex align-items-end">
                                            <button type="button" class="btn btn-outline-danger remove-kalem-row" title="Kalemi Kaldır">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                    
                    <hr class="my-4">
                    
                    <!-- Toplam Alanı -->
                    <div class="row justify-content-end">
                        <div class="col-md-5">
                            <div class="toplam-alan p-3">
                                <h6 class="mb-3 text-center fw-bold"><?= __("total_information") ?></h6>
                                <div class="row mb-2">
                                    <div class="col-6 fw-bold"><?= __("subtotal") ?>:</div>
                                    <div class="col-6 text-end">
                                        <span id="araToplam" class="fw-bold">0,00</span> <?= $paraBirimi ?>
                                    </div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-6 fw-bold"><?= __("discount") ?>:</div>
                                    <div class="col-6">
                                        <input type="text" class="form-control form-control-sm text-end" 
                                               id="indirim_tutari" name="indirim_tutari" 
                                               value="<?= number_format($teklif['indirim_tutari'] ?? 0, 2, ',', '') ?>" 
                                               oninput="genelToplamGuncelle()" placeholder="0,00">
                                    </div>
                                </div>
                                <hr>
                                <div class="row">
                                    <div class="col-6 fw-bold fs-5"><?= __("grand_total") ?>:</div>
                                    <div class="col-6 text-end">
                                        <span id="genelToplam" class="fw-bold fs-5 text-success">0,00 <?= $paraBirimi ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Butonlar -->
            <div class="text-center mb-4">
                <button type="submit" class="btn btn-success btn-lg me-3">
                    <i class="bi bi-save"></i> <?= __("update_quote") ?>
                </button>
                <a href="teklifler.php" class="btn btn-secondary btn-lg">
                    <i class="bi bi-x-circle"></i> <?= __("cancel") ?>
                </a>
            </div>
        </form>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const kalemRowsContainer = document.getElementById('kalemRows');
        const genelToplamElement = document.getElementById('genelToplam');
        const araToplamElement = document.getElementById('araToplam');
        const indirimInput = document.getElementById('indirim_tutari');
        const paraBirimi = "<?= $paraBirimi ?>";

        // Para formatı fonksiyonları
        function formatCurrency(value) {
            let val = parseFloat(value);
            if (isNaN(val)) val = 0;
            return val.toLocaleString('tr-TR', { 
                minimumFractionDigits: 2, 
                maximumFractionDigits: 2 
            });
        }

        function formatInputCurrency(value) {
            let val = parseFloat(value);
            if (isNaN(val)) val = 0;
            return val.toLocaleString('tr-TR', { 
                minimumFractionDigits: 2, 
                maximumFractionDigits: 2,
                useGrouping: true
            });
        }

        // Toplam hesaplama
        function calculateTotal() {
            let total = 0;
            kalemRowsContainer.querySelectorAll('.tutar-input').forEach(input => {
                let valueStr = input.value;
                valueStr = valueStr.replace(/\./g, '');
                valueStr = valueStr.replace(',', '.');
                let value = parseFloat(valueStr) || 0;
                total += value;
            });
            return total;
        }

        function genelToplamGuncelle() {
            const toplam = calculateTotal();
            const indirim = parseFloat(indirimInput.value.replace(/\./g, '').replace(',', '.')) || 0;
            const genelToplam = toplam - indirim;
            
            araToplamElement.textContent = formatCurrency(toplam) + ' ' + paraBirimi;
            genelToplamElement.textContent = formatCurrency(genelToplam) + ' ' + paraBirimi;
        }

        // Satır dinleyicileri
        function addRowListeners(row) {
            const tutarInput = row.querySelector('.tutar-input');
            const removeBtn = row.querySelector('.remove-kalem-row');

            // Tutar input dinleyicisi
            tutarInput.addEventListener('input', function(e) {
                let value = e.target.value.replace(/[^\d,]/g, '');
                e.target.value = value;
                genelToplamGuncelle();
            });

            // Tutar alanından focus çıktığında formatla
            tutarInput.addEventListener('blur', function(e) {
                const numValue = parseFloat(e.target.value.replace(/\./g, '').replace(',', '.'));
                if (!isNaN(numValue) && numValue > 0) {
                    e.target.value = formatInputCurrency(numValue);
                }
            });

            // Kalem kaldırma
            removeBtn.addEventListener('click', function() {
                if (kalemRowsContainer.querySelectorAll('.kalem-row').length > 1) {
                    row.remove();
                    genelToplamGuncelle();
                } else {
                    alert('En az bir kalem bulunmalıdır.');
                }
            });
        }

        // Yeni kalem ekleme
        document.getElementById('addKalemRow').addEventListener('click', function() {
            const firstRow = kalemRowsContainer.querySelector('.kalem-row');
            const newRow = firstRow.cloneNode(true);
            
            // Input değerlerini temizle
            newRow.querySelectorAll('input').forEach(input => input.value = '');
            
            kalemRowsContainer.appendChild(newRow);
            addRowListeners(newRow);
            genelToplamGuncelle();
        });

        // İndirim input dinleyicisi
        indirimInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/[^\d,]/g, '');
            e.target.value = value;
            genelToplamGuncelle();
        });

        // İndirim alanından focus çıktığında formatla
        indirimInput.addEventListener('blur', function(e) {
            const numValue = parseFloat(e.target.value.replace(/\./g, '').replace(',', '.'));
            if (!isNaN(numValue) && numValue >= 0) {
                e.target.value = formatInputCurrency(numValue);
            }
        });

        // Mevcut satırlara dinleyici ekle
        kalemRowsContainer.querySelectorAll('.kalem-row').forEach(row => {
            addRowListeners(row);
        });

        // Sayfa yüklendiğinde mevcut tutar alanlarını formatla
        kalemRowsContainer.querySelectorAll('.tutar-input').forEach(input => {
            if (input.value && input.value !== '0') {
                const numValue = parseFloat(input.value);
                if (!isNaN(numValue)) {
                    input.value = formatInputCurrency(numValue);
                }
            }
        });

        // Sayfa yüklendiğinde toplamı hesapla
        genelToplamGuncelle();

        // Form gönderim kontrolü
        document.getElementById('teklifForm').addEventListener('submit', function(e) {
            const toplam = calculateTotal();
            if (toplam <= 0) {
                e.preventDefault();
                alert('En az bir kalem eklemelisiniz ve toplam tutar 0\'dan büyük olmalıdır.');
                return false;
            }
        });
    });
    </script>
    
    <?php include __DIR__ . '/../includes/footer.php'; ?>
</body>
</html>
