<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/doviz_fonksiyonlari.php';
require_once __DIR__ . '/../includes/yetki_kontrol.php'; // yetki_kontrol_ve_yonlendir fonksiyonu için
require_once __DIR__ . '/../includes/language_functions.php'; // For getCurrentLanguage()
require_once __DIR__ . '/../functions/yardimci_fonksiyonlar.php'; // temizle() ve diğer yardımcı fonksiyonlar için
require_once __DIR__ . '/../functions/log_fonksiyonlari.php'; // Loglama fonksiyonları için EKLENDİ

// Initialize language
$current_language = getCurrentLanguage();
$translation_file = __DIR__ . "/../languages/{$current_language}.php";
$translations = file_exists($translation_file) ? require $translation_file : [];

// Translation function
if (!function_exists('__')) {
    function __($key, $params = []) {
        global $translations;
        $translation = $translations[$key] ?? $key;
        
        // Replace placeholders if any
        if (!empty($params)) {
            foreach ($params as $param => $value) {
                $translation = str_replace(':' . $param, $value, $translation);
            }
        }
        
        return $translation;
    }
}

$sayfa_baslik = __("add_transfer");

// Yetki kontrolü - Sadece belirli yetkilere sahip kullanıcılar erişebilir
yetki_kontrol_ve_yonlendir('transfer_ekle', 'ekle'); // 'transfer_ekle' sayfasının 'ekle' işlemi için yetki kontrolü


// FORM GÖNDERİLDİĞİNDE
if($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST["transfer_ekle"])) {
    if(!csrf_token_dogrula($_POST["csrf_token"])) {
        $_SESSION["hata"] = __("invalid_security_token");
        header("Location: transfer_ekle.php");
        exit();
    }
    
    $kaynak_hesap_id = (int)$_POST["kaynak_hesap"];
    $hedef_hesap_id = (int)$_POST["hedef_hesap"];
    $tutar = (float)str_replace(',', '.', $_POST["tutar"]); // Virgülden sonraki ondalık sayıları destekle
    $tarih = $_POST["tarih"];
    $aciklama = temizle($_POST["aciklama"]);
    $manuel_kur = isset($_POST["manuel_kur"]) ? (float)str_replace(',', '.', $_POST["manuel_kur"]) : 0;
    $use_manual_rate = isset($_POST["use_manual_rate"]) && $_POST["use_manual_rate"] == "1"; // Gizli alandan gelen değer

    try {
        $db->beginTransaction();
        
        // 1. Validasyonlar
        if($kaynak_hesap_id == $hedef_hesap_id) {
            throw new Exception(__("same_account_transfer_error"));
        }
            
        if($tutar <= 0) {
            throw new Exception(__("invalid_amount_error"));
        }
            
        // Hesap bilgilerini çek (hesap_adi eklendi)
        $stmt = $db->prepare("SELECT id, hesap_adi, bakiye, para_birimi, sirket_id FROM hesaplar WHERE (id = ? OR id = ?) AND sirket_id = ?");
        $stmt->execute([$kaynak_hesap_id, $hedef_hesap_id, $_SESSION['aktif_sirket_id'] ?? 0]);
        $hesaplar_assoc = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $hesaplar_map = [];
        foreach ($hesaplar_assoc as $hesap) {
            $hesaplar_map[$hesap["id"]] = $hesap;
        }

        if (!isset($hesaplar_map[$kaynak_hesap_id]) || !isset($hesaplar_map[$hedef_hesap_id])) {
            throw new Exception("Kaynak veya hedef hesap bulunamadı veya bu şirkete ait değil.");
        }

        $kaynak_hesap = $hesaplar_map[$kaynak_hesap_id];
        $hedef_hesap = $hesaplar_map[$hedef_hesap_id];

        $kaynak_bakiye = $kaynak_hesap["bakiye"]; // İşlem öncesi kaynak bakiye
        $hedef_bakiye = $hedef_hesap["bakiye"];   // İşlem öncesi hedef bakiye

        $kaynak_para_birimi = $kaynak_hesap["para_birimi"];
        $hedef_para_birimi = $hedef_hesap["para_birimi"];

        // Transfer edilecek tutarların başlangıç değerleri
        $transfer_edilecek_tutar_kaynak = $tutar; // Kaynak hesaptan düşülecek miktar (giriş tutarı)
        $transfer_edilecek_tutar_hedef = $tutar;  // Hedef hesaba eklenecek miktar (çıkış tutarı)
        $exchange_rate = 1.0; // Kur değeri

        // Para birimleri farklıysa döviz kuru dönüşümü yap
        if ($kaynak_para_birimi !== $hedef_para_birimi) {
            if ($use_manual_rate && $manuel_kur > 0) { // Manuel kur kullanılıyorsa
                $exchange_rate = $manuel_kur;
                $_SESSION["mesaj"] = "Manuel kur (" . number_format($exchange_rate, 4, ',', '.') . ") kullanıldı.";
            } else { // Otomatik kur alımı
                $exchange_rate = get_exchange_rate($kaynak_para_birimi, $hedef_para_birimi);
                if ($exchange_rate === false || $exchange_rate <= 0) {
                    throw new Exception("Otomatik döviz kuru alınamadı veya geçersiz. Lütfen manuel kur giriniz.");
                }
                $_SESSION["mesaj"] = "Otomatik kur (" . number_format($exchange_rate, 4, ',', '.') . ") kullanıldı.";
            }
            $transfer_edilecek_tutar_hedef = $tutar * $exchange_rate;
            $_SESSION["mesaj"] .= " " . number_format($tutar, 2, ',', '.') . " " . $kaynak_para_birimi . " = " . number_format($transfer_edilecek_tutar_hedef, 2, ',', '.') . " " . $hedef_para_birimi;
        }

        // 2. Kaynak hesap bakiyesini kontrol et
        if($kaynak_bakiye < $transfer_edilecek_tutar_kaynak) {
            throw new Exception("Yetersiz bakiye! Mevcut bakiye: ".number_format($kaynak_bakiye,2, ',', '.')." " . $kaynak_para_birimi);
        }
        
        // 3. Hareketleri kaydet
        // Kaynaktan çıkış hareketi
        $stmt_cikis = $db->prepare("INSERT INTO hesap_hareketleri 
            (hesap_id, tur, tutar, aciklama, tarih, baglanti_id, sirket_id) 
            VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt_cikis->execute([$kaynak_hesap_id, 'cikis', $transfer_edilecek_tutar_kaynak, $aciklama, $tarih, null, $_SESSION['aktif_sirket_id'] ?? 0]);
        $kaynak_hareket_id = $db->lastInsertId();

        // Hedefe giriş hareketi
        $stmt_giris = $db->prepare("INSERT INTO hesap_hareketleri 
            (hesap_id, tur, tutar, aciklama, tarih, baglanti_id, sirket_id) 
            VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt_giris->execute([$hedef_hesap_id, 'giris', $transfer_edilecek_tutar_hedef, $aciklama, $tarih, null, $_SESSION['aktif_sirket_id'] ?? 0]);
        $hedef_hareket_id = $db->lastInsertId();

        // Her iki hareketi birbirine bağla (baglanti_id güncellemesi)
        $db->prepare("UPDATE hesap_hareketleri SET baglanti_id = ? WHERE id = ?")
           ->execute([$hedef_hareket_id, $kaynak_hareket_id]);
        $db->prepare("UPDATE hesap_hareketleri SET baglanti_id = ? WHERE id = ?")
           ->execute([$kaynak_hareket_id, $hedef_hareket_id]);
        
        // 4. Bakiyeleri güncelle
        $db->prepare("UPDATE hesaplar SET bakiye = bakiye - ? WHERE id = ?")
           ->execute([$transfer_edilecek_tutar_kaynak, $kaynak_hesap_id]);
           
        $db->prepare("UPDATE hesaplar SET bakiye = bakiye + ? WHERE id = ?")
           ->execute([$transfer_edilecek_tutar_hedef, $hedef_hesap_id]);

        // 5. Loglama İşlemleri EKLENDİ
        // Kaynak hesaptan çıkışın loglanması
        sistem_log_ekle(
            $db,
            'transfer_cikis',
            'hesap_hareketleri',
            $kaynak_hareket_id, 
            [
                'hesap_adi' => $kaynak_hesap['hesap_adi'],
                'eski_bakiye' => (float)$kaynak_bakiye,
                'transfer_tutari' => (float)$transfer_edilecek_tutar_kaynak,
                'para_birimi' => $kaynak_para_birimi
            ], 
            [
                'yeni_bakiye' => (float)($kaynak_bakiye - $transfer_edilecek_tutar_kaynak)
            ], 
            "Hesaptan para çıkışı: " . $kaynak_hesap['hesap_adi'] . " -> " . $hedef_hesap['hesap_adi'] . " için " . number_format($transfer_edilecek_tutar_kaynak, 2, ',', '.') . " " . $kaynak_para_birimi
        );

        // Hedef hesaba girişin loglanması
        sistem_log_ekle(
            $db,
            'transfer_giris',
            'hesap_hareketleri',
            $hedef_hareket_id, 
            [
                'hesap_adi' => $hedef_hesap['hesap_adi'],
                'eski_bakiye' => (float)$hedef_bakiye,
                'transfer_tutari' => (float)$transfer_edilecek_tutar_hedef,
                'para_birimi' => $hedef_para_birimi
            ], 
            [
                'yeni_bakiye' => (float)($hedef_bakiye + $transfer_edilecek_tutar_hedef)
            ], 
            "Hesaba para girişi: " . $kaynak_hesap['hesap_adi'] . " -> " . $hedef_hesap['hesap_adi'] . " için " . number_format($transfer_edilecek_tutar_hedef, 2, ',', '.') . " " . $hedef_para_birimi
        );

        // Genel transfer işleminin loglanması
        sistem_log_ekle(
            $db,
            'transfer',
            'hesaplar', 
            null, 
            [
                'kaynak_hesap_id' => $kaynak_hesap_id,
                'kaynak_hesap_adi' => $kaynak_hesap['hesap_adi'],
                'hedef_hesap_id' => $hedef_hesap_id,
                'hedef_hesap_adi' => $hedef_hesap['hesap_adi'],
                'transfer_tutari_kaynak' => (float)$transfer_edilecek_tutar_kaynak,
                'para_birimi_kaynak' => $kaynak_para_birimi,
                'transfer_tutari_hedef' => (float)$transfer_edilecek_tutar_hedef,
                'para_birimi_hedef' => $hedef_para_birimi,
                'manuel_kur_kullanildi' => $use_manual_rate,
                'kullanilan_kur' => (float)$exchange_rate,
                'aciklama' => $aciklama
            ], 
            null, 
            "Hesaplar arası para transferi gerçekleşti: " . $kaynak_hesap['hesap_adi'] . " (" . $kaynak_para_birimi . ") -> " . $hedef_hesap['hesap_adi'] . " (" . $hedef_para_birimi . ") Tutar: " . number_format($tutar, 2, ',', '.') . " " . $kaynak_para_birimi
        );
        
        $db->commit();
        $_SESSION["mesaj"] = $_SESSION["mesaj"] ?? "Transfer başarıyla tamamlandı!";
    } catch(Exception $e) {
        $db->rollBack();
        $_SESSION["hata"] = $e->getMessage();
        error_log("Para Transferi Hatası: " . $e->getMessage()); // Hataları logla
    }
    header("Location: transfer_ekle.php"); exit();
}

// HESAPLARI GETİR
try {
    $aktif_sirket_id = $_SESSION['aktif_sirket_id'] ?? 0;
    // Sadece aktif şirkete ait hesaplar çekilecek
    $hesaplar = $db->prepare("SELECT id, hesap_adi, bakiye, para_birimi FROM hesaplar WHERE sirket_id = ? ORDER BY hesap_adi");
    $hesaplar->execute([$aktif_sirket_id]);
    $hesaplar = $hesaplar->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    $_SESSION["hata"] = "Hesaplar yüklenirken bir hata oluştu: " . $e->getMessage();
    $hesaplar = []; // Hata durumunda boş dizi döndür
    error_log("Hesap yükleme hatası: " . $e->getMessage());
}

$csrf_token = csrf_token_olustur();
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Odjoo Yönetim - <?= temizle($sayfa_baslik) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
</head>
<body>
    <?php include __DIR__ . '/../includes/navbar.php'; ?>

    <div class="container-xl mt-4">
        <div class="row">
            <div class="col mx-auto">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-arrow-left-right me-2"></i> <?= __("add_transfer") ?></h5>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            <input type="hidden" name="use_manual_rate" id="use_manual_rate_hidden" value="0">
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="kaynak_hesap_select" class="form-label"><?= __("source_account") ?></label>
                                    <select name="kaynak_hesap" class="form-select" id="kaynak_hesap_select" required>
                                        <?php if (empty($hesaplar)): ?>
                                            <option value=""><?= __("select_source_account") ?></option>
                                        <?php else: ?>
                                            <?php foreach($hesaplar as $hesap): ?>
                                            <option value="<?= $hesap["id"] ?>" data-currency="<?= temizle($hesap["para_birimi"]) ?>">
                                                <?= temizle($hesap["hesap_adi"]) ?> 
                                                (<?= number_format($hesap["bakiye"], 2, ',', '.') ?> <?= temizle($hesap["para_birimi"]) ?>)
                                            </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <small class="text-muted" id="kaynak_para_birimi_info"></small>
                                </div>
                                <div class="col-md-6">
                                    <label for="hedef_hesap_select" class="form-label"><?= __("target_account") ?></label>
                                    <select name="hedef_hesap" class="form-select" id="hedef_hesap_select" required>
                                        <?php if (empty($hesaplar)): ?>
                                            <option value=""><?= __("select_target_account") ?></option>
                                        <?php else: ?>
                                            <?php foreach($hesaplar as $hesap): ?>
                                            <option value="<?= $hesap["id"] ?>" data-currency="<?= temizle($hesap["para_birimi"]) ?>">
                                                <?= temizle($hesap["hesap_adi"]) ?> 
                                                (<?= number_format($hesap["bakiye"], 2, ',', '.') ?> <?= temizle($hesap["para_birimi"]) ?>)
                                            </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <small class="text-muted" id="hedef_para_birimi_info"></small>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="tutar_input" class="form-label"><?= __("transfer_amount") ?> (<span id="tutar_para_birimi"></span>)</label>
                                <input type="text" inputmode="decimal" step="0.01" min="0.01" 
                                       name="tutar" class="form-control" id="tutar_input" required
                                       placeholder="<?= __("transfer_amount_placeholder") ?>">
                            </div>

                            <div class="mb-3" id="kur_bilgisi_alani" style="display: none;">
                                <label for="manuel_kur_input" class="form-label"><?= __("exchange_rate") ?></label>
                                <input type="text" inputmode="decimal" step="0.0001" min="0.0001"
                                       name="manuel_kur" class="form-control" id="manuel_kur_input"
                                       placeholder="<?= __("exchange_rate_placeholder") ?>" readonly>
                                <small class="text-muted mt-2" id="donusum_bilgisi">
                                    <?= __("automatic_exchange_rate_info") ?>
                                </small>
                            </div>

                            <div class="mb-3 form-check" id="manuel_kur_check_div">
                                <input class="form-check-input" type="checkbox" id="manuel_kur_checkbox">
                                <label class="form-check-label" for="manuel_kur_checkbox">
                                    <?= __("use_manual_rate") ?>
                                </label>
                            </div>
                            
                            <div class="mb-3">
                                <label for="tarih" class="form-label"><?= __("transfer_date") ?> <span class="text-danger">*</span></label>
                                <input type="date" name="tarih" id="tarih"
                                       value="<?= date('Y-m-d') ?>" class="form-control" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="aciklama" class="form-label"><?= __("transfer_description") ?></label>
                                <textarea name="aciklama" id="aciklama" class="form-control" rows="2" placeholder="<?= __("transfer_description_placeholder") ?>"></textarea>
                            </div>
                            
                            <button type="submit" name="transfer_ekle" 
                                    class="btn btn-primary w-100 mt-3">
                                <i class="bi bi-send-check me-2"></i> <?= __("save") ?>
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
	<?php include __DIR__ . "/../includes/footer.php"; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const kaynakHesapSelect = document.getElementById('kaynak_hesap_select');
        const hedefHesapSelect = document.getElementById('hedef_hesap_select');
        const tutarInput = document.getElementById('tutar_input');
        const tutarParaBirimiSpan = document.getElementById('tutar_para_birimi');
        const donusumBilgisi = document.getElementById('donusum_bilgisi');
        const kaynakParaBirimiInfo = document.getElementById('kaynak_para_birimi_info');
        const hedefParaBirimiInfo = document.getElementById('hedef_para_birimi_info');
        const manuelKurCheckbox = document.getElementById('manuel_kur_checkbox');
        const kurBilgisiAlani = document.getElementById('kur_bilgisi_alani'); // Yeni ID
        const manuelKurInput = document.getElementById('manuel_kur_input');
        const useManualRateHiddenInput = document.getElementById('use_manual_rate_hidden'); // Gizli input

        // PHP'den gelen hesaplar verisini doğrudan kullan
        const hesaplar = <?= json_encode($hesaplar, JSON_UNESCAPED_UNICODE) ?>;

        // Tutar input alanına sadece sayı ve virgül/nokta girişi kısıtlaması
        tutarInput.addEventListener('keypress', function(event) {
            if ((event.key < '0' || event.key > '9') && event.key !== '.' && event.key !== ',') {
                event.preventDefault();
            }
            // Sadece bir nokta veya virgül olmalı
            if ((event.key === '.' || event.key === ',') && (this.value.includes('.') || this.value.includes(','))) {
                event.preventDefault();
            }
        });
        tutarInput.addEventListener('paste', function(event) {
            const pasteData = event.clipboardData.getData('text');
            if (!/^\d*[\.,]?\d*$/.test(pasteData)) {
                event.preventDefault();
            }
        });
        tutarInput.addEventListener('input', function() {
            this.value = this.value.replace(',', '.'); // Virgülü noktaya çevir
        });

        manuelKurInput.addEventListener('keypress', function(event) {
            if ((event.key < '0' || event.key > '9') && event.key !== '.' && event.key !== ',') {
                event.preventDefault();
            }
            if ((event.key === '.' || event.key === ',') && (this.value.includes('.') || this.value.includes(','))) {
                event.preventDefault();
            }
        });
        manuelKurInput.addEventListener('paste', function(event) {
            const pasteData = event.clipboardData.getData('text');
            if (!/^\d*[\.,]?\d*$/.test(pasteData)) {
                event.preventDefault();
            }
        });
        manuelKurInput.addEventListener('input', function() {
            this.value = this.value.replace(',', '.'); // Virgülü noktaya çevir
        });


        function updateCurrencyInfo() {
            const selectedKaynakHesapId = kaynakHesapSelect.value;
            const selectedHedefHesapId = hedefHesapSelect.value;

            const kaynakHesap = hesaplar.find(h => h.id == selectedKaynakHesapId);
            const hedefHesap = hesaplar.find(h => h.id == selectedHedefHesapId);

            const kaynakParaBirimi = kaynakHesap ? kaynakHesap.para_birimi : '';
            const hedefParaBirimi = hedefHesap ? hedefHesap.para_birimi : '';

            kaynakParaBirimiInfo.textContent = `Para Birimi: ${kaynakParaBirimi}`;
            hedefParaBirimiInfo.textContent = `Para Birimi: ${hedefParaBirimi}`;

            tutarParaBirimiSpan.textContent = kaynakParaBirimi;

            const isCurrenciesDifferent = (kaynakParaBirimi && hedefParaBirimi && kaynakParaBirimi !== hedefParaBirimi);
            
            // Para birimleri farklıysa kur alanını göster, değilse gizle ve manuel kur checkbox'ını devre dışı bırak
            if (isCurrenciesDifferent) {
                kurBilgisiAlani.style.display = 'block'; // Kur bilgi alanını göster
                manuelKurCheckbox.disabled = false;
            } else {
                kurBilgisiAlani.style.display = 'none'; // Kur bilgi alanını gizle
                manuelKurCheckbox.checked = false; // Manuel kur seçimini kaldır
                manuelKurCheckbox.disabled = true; // Manuel kur checkbox'ını devre dışı bırak
                manuelKurInput.value = ''; // Manuel kur değerini temizle
                manuelKurInput.readOnly = true; // Manuel kur inputunu salt okunur yap
                donusumBilgisi.textContent = ''; // Dönüşüm bilgisini temizle
                useManualRateHiddenInput.value = '0'; // Gizli alanı sıfırla
                return; // Eğer para birimleri aynıysa daha fazla işlem yapma
            }

            // Manuel kur checkbox'ının durumuna göre input'u yönet
            if (manuelKurCheckbox.checked) {
                manuelKurInput.readOnly = false; // Manuel kur girilebilir yap
                useManualRateHiddenInput.value = '1'; // Manuel kur kullanıldığını belirt
            } else {
                manuelKurInput.readOnly = true; // Otomatik kur için salt okunur yap
                useManualRateHiddenInput.value = '0'; // Otomatik kur kullanıldığını belirt
            }

            const tutar = parseFloat(tutarInput.value);
            const useManualRate = manuelKurCheckbox.checked;

            if (useManualRate) {
                const manuelKur = parseFloat(manuelKurInput.value);
                if (!isNaN(manuelKur) && manuelKur > 0) {
                    const donusenTutar = tutar * manuelKur;
                    donusumBilgisi.innerHTML = `Yaklaşık: <strong>${donusenTutar.toFixed(2)} ${hedefParaBirimi}</strong> (Manuel Kur: ${manuelKur.toFixed(4)})`;
                    donusumBilgisi.classList.remove('text-danger');
                    donusumBilgisi.classList.add('text-muted');
                } else {
                    donusumBilgisi.textContent = 'Manuel kur geçerli değil. Lütfen geçerli bir kur giriniz.';
                    donusumBilgisi.classList.remove('text-muted');
                    donusumBilgisi.classList.add('text-danger');
                }
            } else {
                // Otomatik kur alımı
                if (!isNaN(tutar) && tutar > 0) {
                    fetch(`../ajax/get_exchange_rate.php?from=${kaynakParaBirimi}&to=${hedefParaBirimi}`)
                        .then(response => {
                            if (!response.ok) {
                                throw new Error('Network response was not ok ' + response.statusText);
                            }
                            return response.json();
                        })
                        .then(data => {
                            if (data.success && !isNaN(parseFloat(data.rate)) && parseFloat(data.rate) > 0) {
                                const kur = parseFloat(data.rate);
                                const donusenTutar = tutar * kur;
                                donusumBilgisi.innerHTML = `Yaklaşık: <strong>${donusenTutar.toFixed(2)} ${hedefParaBirimi}</strong> (Otomatik Kur: ${kur.toFixed(4)})`;
                                donusumBilgisi.classList.remove('text-danger');
                                donusumBilgisi.classList.add('text-muted');
                                manuelKurInput.value = kur.toFixed(4); // Otomatik kuru manuel alana da yaz
                            } else {
                                donusumBilgisi.textContent = `Döviz kuru alınamadı: ${data.message || 'Bilinmeyen Hata'}`;
                                donusumBilgisi.classList.remove('text-muted');
                                donusumBilgisi.classList.add('text-danger');
                                manuelKurInput.value = ''; // Hata durumunda manuel kuru temizle
                            }
                        })
                        .catch(error => {
                            console.error('Döviz kuru AJAX hatası:', error);
                            donusumBilgisi.textContent = 'Döviz kuru alınırken bir hata oluştu.';
                            donusumBilgisi.classList.remove('text-muted');
                            donusumBilgisi.classList.add('text-danger');
                            manuelKurInput.value = ''; // Hata durumunda manuel kuru temizle
                        });
                } else {
                    donusumBilgisi.textContent = 'Para birimleri farklı. Lütfen tutarı giriniz.';
                    donusumBilgisi.classList.remove('text-danger');
                    donusumBilgisi.classList.add('text-muted');
                    manuelKurInput.value = ''; // Tutar girilmediğinde manuel kuru temizle
                }
            }
        }

        kaynakHesapSelect.addEventListener('change', updateCurrencyInfo);
        hedefHesapSelect.addEventListener('change', updateCurrencyInfo);
        tutarInput.addEventListener('input', updateCurrencyInfo);
        manuelKurInput.addEventListener('input', updateCurrencyInfo);
        
        manuelKurCheckbox.addEventListener('change', updateCurrencyInfo); // Sadece updateCurrencyInfo'yu çağır

        // Sayfa yüklendiğinde ilk bilgileri göster
        updateCurrencyInfo();
    </script>
</body>
</html>
